<?php
/**
 * Cart Page Template
 *
 * Custom styled cart page for AI Theme
 *
 * @package AI_Theme
 */

defined('ABSPATH') || exit;

// Clear WooCommerce notices on cart page (they appear as duplicate banners)
wc_clear_notices();

$cart_is_empty = WC()->cart->is_empty();
?>

<div class="ai-cart-page py-8 md:py-12" data-theme="<?php echo esc_attr(get_option('ai_theme_style', 'modern')); ?>">
    <div class="ai-container">

        <!-- Page Header -->
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900 mb-2"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.title', 'Cart')); ?></h1>
            <nav class="flex text-sm text-gray-500">
                <a href="<?php echo esc_url(home_url('/')); ?>" class="hover:text-primary transition-colors"><?php echo esc_html(ai_theme_get_localized_text('common.home', 'Home')); ?></a>
                <span class="mx-2">/</span>
                <span class="text-gray-900"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.title', 'Cart')); ?></span>
            </nav>
        </div>

        <?php if ($cart_is_empty) : ?>
            <!-- Empty Cart -->
            <div class="text-center py-16 bg-gray-50 rounded-2xl">
                <div class="w-24 h-24 mx-auto mb-6 bg-gray-200 rounded-full flex items-center justify-center">
                    <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                    </svg>
                </div>
                <h2 class="text-2xl font-semibold text-gray-900 mb-2"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.emptyTitle', 'Your cart is empty')); ?></h2>
                <p class="text-gray-600 mb-8 max-w-md mx-auto"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.emptyText', 'Looks like you haven\'t added any items to your cart yet. Browse our catalog to find something you like.')); ?></p>
                <a href="<?php echo esc_url(apply_filters('woocommerce_return_to_shop_redirect', wc_get_page_permalink('shop'))); ?>" class="inline-flex items-center gap-2 px-6 py-3 bg-primary text-white font-semibold rounded-lg hover:opacity-90 transition-opacity">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                    </svg>
                    <?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.continueShopping', 'Continue Shopping')); ?>
                </a>
            </div>

        <?php else : ?>

            <form class="woocommerce-cart-form" action="<?php echo esc_url(wc_get_cart_url()); ?>" method="post">
                <?php do_action('woocommerce_before_cart_table'); ?>

                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Cart Items -->
                    <div class="lg:col-span-2">
                        <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
                            <!-- Header -->
                            <div class="hidden md:grid grid-cols-12 gap-4 px-6 py-4 bg-gray-50 border-b border-gray-100 text-sm font-medium text-gray-500">
                                <div class="col-span-6"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.product', 'Product')); ?></div>
                                <div class="col-span-2 text-center"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.price', 'Price')); ?></div>
                                <div class="col-span-2 text-center"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.quantity', 'Qty')); ?></div>
                                <div class="col-span-2 text-right"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.subtotal', 'Subtotal')); ?></div>
                            </div>

                            <!-- Items -->
                            <div class="divide-y divide-gray-100">
                                <?php do_action('woocommerce_before_cart_contents'); ?>

                                <?php foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) :
                                    $_product = apply_filters('woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key);
                                    $product_id = apply_filters('woocommerce_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key);
                                    $product_name = apply_filters('woocommerce_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key);

                                    if ($_product && $_product->exists() && $cart_item['quantity'] > 0 && apply_filters('woocommerce_cart_item_visible', true, $cart_item, $cart_item_key)) :
                                        $product_permalink = apply_filters('woocommerce_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink($cart_item) : '', $cart_item, $cart_item_key);
                                ?>
                                <div class="<?php echo esc_attr(apply_filters('woocommerce_cart_item_class', 'cart_item', $cart_item, $cart_item_key)); ?> grid grid-cols-12 gap-4 items-center p-4 md:p-6">
                                    <!-- Product -->
                                    <div class="col-span-12 md:col-span-6 flex items-center gap-4">
                                        <!-- Remove -->
                                        <?php
                                        echo apply_filters(
                                            'woocommerce_cart_item_remove_link',
                                            sprintf(
                                                '<a href="%s" class="remove flex-shrink-0 w-8 h-8 flex items-center justify-center text-gray-400 hover:text-red-500 hover:bg-red-50 rounded-lg transition-colors" aria-label="%s" data-product_id="%s" data-product_sku="%s"><svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg></a>',
                                                esc_url(wc_get_cart_remove_url($cart_item_key)),
                                                esc_html__('Remove this item', 'woocommerce'),
                                                esc_attr($product_id),
                                                esc_attr($_product->get_sku())
                                            ),
                                            $cart_item_key
                                        );
                                        ?>

                                        <!-- Thumbnail -->
                                        <div class="flex-shrink-0 w-20 h-20 bg-gray-100 rounded-xl overflow-hidden">
                                            <?php
                                            $thumbnail = apply_filters('woocommerce_cart_item_thumbnail', $_product->get_image('woocommerce_thumbnail', array('class' => 'w-full h-full object-cover')), $cart_item, $cart_item_key);

                                            if (!$product_permalink) {
                                                echo $thumbnail;
                                            } else {
                                                printf('<a href="%s">%s</a>', esc_url($product_permalink), $thumbnail);
                                            }
                                            ?>
                                        </div>

                                        <!-- Name & Meta -->
                                        <div class="flex-1 min-w-0">
                                            <?php
                                            if (!$product_permalink) {
                                                echo '<h3 class="font-medium text-gray-900 truncate">' . wp_kses_post($product_name) . '</h3>';
                                            } else {
                                                echo '<a href="' . esc_url($product_permalink) . '" class="font-medium text-gray-900 hover:text-primary truncate block">' . wp_kses_post($product_name) . '</a>';
                                            }

                                            do_action('woocommerce_after_cart_item_name', $cart_item, $cart_item_key);

                                            // Meta data
                                            echo '<div class="text-sm text-gray-500 mt-1">';
                                            echo wc_get_formatted_cart_item_data($cart_item);

                                            // Backorder notification
                                            if ($_product->backorders_require_notification() && $_product->is_on_backorder($cart_item['quantity'])) {
                                                echo '<p class="text-amber-600">' . esc_html(apply_filters('woocommerce_cart_item_backorder_notification', __('Available on backorder', 'woocommerce'), $product_id)) . '</p>';
                                            }
                                            echo '</div>';
                                            ?>

                                            <!-- Mobile Price -->
                                            <div class="md:hidden mt-2 text-sm text-gray-500">
                                                <?php
                                                echo apply_filters('woocommerce_cart_item_price', WC()->cart->get_product_price($_product), $cart_item, $cart_item_key);
                                                ?>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Price (Desktop) -->
                                    <div class="hidden md:block col-span-2 text-center text-gray-600">
                                        <?php echo apply_filters('woocommerce_cart_item_price', WC()->cart->get_product_price($_product), $cart_item, $cart_item_key); ?>
                                    </div>

                                    <!-- Quantity -->
                                    <div class="col-span-6 md:col-span-2">
                                        <?php
                                        if ($_product->is_sold_individually()) {
                                            $min_quantity = 1;
                                            $max_quantity = 1;
                                        } else {
                                            $min_quantity = 0;
                                            $max_quantity = $_product->get_max_purchase_quantity();
                                        }

                                        $product_quantity = woocommerce_quantity_input(
                                            array(
                                                'input_name'   => "cart[{$cart_item_key}][qty]",
                                                'input_value'  => $cart_item['quantity'],
                                                'max_value'    => $max_quantity,
                                                'min_value'    => $min_quantity,
                                                'product_name' => $product_name,
                                                'classes'      => array('ai-qty-input'),
                                            ),
                                            $_product,
                                            false
                                        );

                                        echo apply_filters('woocommerce_cart_item_quantity', $product_quantity, $cart_item_key, $cart_item);
                                        ?>
                                    </div>

                                    <!-- Subtotal -->
                                    <div class="col-span-6 md:col-span-2 text-right font-semibold text-gray-900">
                                        <?php echo apply_filters('woocommerce_cart_item_subtotal', WC()->cart->get_product_subtotal($_product, $cart_item['quantity']), $cart_item, $cart_item_key); ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                <?php endforeach; ?>

                                <?php do_action('woocommerce_cart_contents'); ?>
                            </div>

                            <!-- Actions -->
                            <div class="px-6 py-4 bg-gray-50 border-t border-gray-100 flex flex-wrap items-center justify-between gap-4">
                                <div class="flex items-center gap-2">
                                    <?php if (wc_coupons_enabled()) : ?>
                                    <input type="text" name="coupon_code" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent text-sm" id="coupon_code" value="" placeholder="<?php echo esc_attr(ai_theme_get_localized_text('ecommerce.cart.couponPlaceholder', 'Coupon code')); ?>" />
                                    <button type="submit" class="px-4 py-2 bg-gray-200 hover:bg-gray-300 text-gray-700 font-medium rounded-lg transition-colors text-sm" name="apply_coupon" value="<?php echo esc_attr(ai_theme_get_localized_text('ecommerce.cart.applyCoupon', 'Apply')); ?>"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.applyCoupon', 'Apply')); ?></button>
                                    <?php do_action('woocommerce_cart_coupon'); ?>
                                    <?php endif; ?>
                                </div>

                                <button type="submit" class="px-4 py-2 bg-gray-200 hover:bg-gray-300 text-gray-700 font-medium rounded-lg transition-colors text-sm flex items-center gap-2" name="update_cart" value="<?php echo esc_attr(ai_theme_get_localized_text('ecommerce.cart.updateCart', 'Update cart')); ?>">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                                    </svg>
                                    <?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.updateCart', 'Update cart')); ?>
                                </button>

                                <?php do_action('woocommerce_cart_actions'); ?>
                                <?php wp_nonce_field('woocommerce-cart', 'woocommerce-cart-nonce'); ?>
                            </div>
                        </div>

                        <?php do_action('woocommerce_after_cart_contents'); ?>
                    </div>

                    <!-- Cart Totals -->
                    <div class="lg:col-span-1">
                        <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-6 sticky top-24">
                            <h2 class="text-xl font-bold text-gray-900 mb-6"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.cartTotals', 'Cart totals')); ?></h2>

                            <?php do_action('woocommerce_before_cart_totals'); ?>

                            <div class="space-y-4">
                                <!-- Subtotal -->
                                <div class="flex items-center justify-between py-3 border-b border-gray-100">
                                    <span class="text-gray-600"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.itemsLabel', 'Items:')); ?></span>
                                    <span class="font-medium text-gray-900"><?php wc_cart_totals_subtotal_html(); ?></span>
                                </div>

                                <!-- Coupons -->
                                <?php foreach (WC()->cart->get_coupons() as $code => $coupon) : ?>
                                <div class="flex items-center justify-between py-3 border-b border-gray-100 cart-discount coupon-<?php echo esc_attr(sanitize_title($code)); ?>">
                                    <span class="text-gray-600"><?php wc_cart_totals_coupon_label($coupon); ?></span>
                                    <span class="font-medium text-green-600"><?php wc_cart_totals_coupon_html($coupon); ?></span>
                                </div>
                                <?php endforeach; ?>

                                <!-- Shipping -->
                                <?php if (WC()->cart->needs_shipping() && WC()->cart->show_shipping()) : ?>
                                    <?php do_action('woocommerce_cart_totals_before_shipping'); ?>
                                    <div class="py-3 border-b border-gray-100">
                                        <?php wc_cart_totals_shipping_html(); ?>
                                    </div>
                                    <?php do_action('woocommerce_cart_totals_after_shipping'); ?>
                                <?php elseif (WC()->cart->needs_shipping() && 'yes' === get_option('woocommerce_enable_shipping_calc')) : ?>
                                <div class="py-3 border-b border-gray-100">
                                    <p class="text-sm text-gray-500"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.shippingCalculated', 'Shipping calculated at checkout.')); ?></p>
                                </div>
                                <?php endif; ?>

                                <!-- Fees -->
                                <?php foreach (WC()->cart->get_fees() as $fee) : ?>
                                <div class="flex items-center justify-between py-3 border-b border-gray-100">
                                    <span class="text-gray-600"><?php echo esc_html($fee->name); ?></span>
                                    <span class="font-medium text-gray-900"><?php wc_cart_totals_fee_html($fee); ?></span>
                                </div>
                                <?php endforeach; ?>

                                <!-- Tax -->
                                <?php if (wc_tax_enabled() && !WC()->cart->display_prices_including_tax()) :
                                    $taxable_address = WC()->customer->get_taxable_address();
                                    $estimated_text  = '';

                                    if (WC()->customer->is_customer_outside_base() && !WC()->customer->has_calculated_shipping()) {
                                        /* translators: %s location. */
                                        $estimated_text = sprintf(' <small>' . esc_html__('(estimated for %s)', 'woocommerce') . '</small>', WC()->countries->estimated_for_prefix($taxable_address[0]) . WC()->countries->countries[$taxable_address[0]]);
                                    }

                                    if ('itemized' === get_option('woocommerce_tax_total_display')) :
                                        foreach (WC()->cart->get_tax_totals() as $code => $tax_total) :
                                ?>
                                <div class="flex items-center justify-between py-3 border-b border-gray-100">
                                    <span class="text-gray-600"><?php echo esc_html($tax_total->label) . $estimated_text; ?></span>
                                    <span class="font-medium text-gray-900"><?php echo wp_kses_post($tax_total->formatted_amount); ?></span>
                                </div>
                                <?php endforeach;
                                    else :
                                ?>
                                <div class="flex items-center justify-between py-3 border-b border-gray-100">
                                    <span class="text-gray-600"><?php echo esc_html(WC()->countries->tax_or_vat()) . $estimated_text; ?></span>
                                    <span class="font-medium text-gray-900"><?php wc_cart_totals_taxes_total_html(); ?></span>
                                </div>
                                <?php endif; ?>
                                <?php endif; ?>

                                <?php do_action('woocommerce_cart_totals_before_order_total'); ?>

                                <!-- Total -->
                                <div class="flex items-center justify-between pt-4 border-t-2 border-gray-200">
                                    <span class="text-lg font-semibold text-gray-900"><?php echo esc_html(ai_theme_get_localized_text('ecommerce.cart.total', 'Total:')); ?></span>
                                    <span class="text-2xl font-bold text-primary"><?php wc_cart_totals_order_total_html(); ?></span>
                                </div>

                                <?php do_action('woocommerce_cart_totals_after_order_total'); ?>
                            </div>

                            <div class="mt-6">
                                <?php do_action('woocommerce_proceed_to_checkout'); ?>
                            </div>

                            <!-- Continue Shopping -->
                            <div class="mt-4 text-center">
                                <a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>" class="text-sm text-gray-500 hover:text-primary transition-colors">
                                    <?php echo esc_html('← ' . ai_theme_get_localized_text('ecommerce.cart.continueShopping', 'Continue shopping')); ?>
                                </a>
                            </div>

                            <?php do_action('woocommerce_after_cart_totals'); ?>
                        </div>
                    </div>
                </div>

                <?php do_action('woocommerce_after_cart_table'); ?>
            </form>

            <?php do_action('woocommerce_before_cart_collaterals'); ?>

            <div class="cart-collaterals mt-8">
                <?php
                /**
                 * Cart collaterals hook.
                 *
                 * @hooked woocommerce_cross_sell_display - 10
                 */
                do_action('woocommerce_cart_collaterals');
                ?>
            </div>

        <?php endif; ?>
    </div>
</div>

<?php do_action('woocommerce_after_cart'); ?>

<style>
/* Custom quantity input */
.ai-qty-input input[type="number"] {
    width: 100%;
    max-width: 80px;
    text-align: center;
    padding: 0.5rem;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    font-size: 0.875rem;
}
.ai-qty-input input[type="number"]:focus {
    outline: none;
    border-color: var(--color-primary, #3b82f6);
    box-shadow: 0 0 0 2px rgba(59, 130, 246, 0.1);
}
/* Hide spinners */
.ai-qty-input input[type="number"]::-webkit-inner-spin-button,
.ai-qty-input input[type="number"]::-webkit-outer-spin-button {
    -webkit-appearance: none;
    margin: 0;
}
.ai-qty-input input[type="number"] {
    -moz-appearance: textfield;
}
/* Checkout button */
.wc-proceed-to-checkout .checkout-button {
    display: block;
    width: 100%;
    padding: 1rem 1.5rem;
    background-color: var(--color-primary, #3b82f6);
    color: white;
    font-weight: 600;
    text-align: center;
    border-radius: 0.75rem;
    transition: opacity 0.2s;
}
.wc-proceed-to-checkout .checkout-button:hover {
    opacity: 0.9;
}
/* Shipping calculator styling */
.shipping-calculator-form {
    margin-top: 1rem;
    padding: 1rem;
    background: #f9fafb;
    border-radius: 0.5rem;
}
.shipping-calculator-form input,
.shipping-calculator-form select {
    width: 100%;
    padding: 0.5rem;
    margin-bottom: 0.5rem;
    border: 1px solid #e5e7eb;
    border-radius: 0.375rem;
}
</style>
