<?php
/**
 * Product Archive / Category Template - Uses AI Theme Styling
 *
 * This template renders WooCommerce product archives and categories
 * with our custom AI Theme styling including breadcrumbs.
 *
 * @package AI_Theme
 */

if (!defined('ABSPATH')) {
    exit;
}

get_header();

// Get current category info
$current_term = get_queried_object();
$is_category = is_product_category();

// Get AI-generated texts with fallbacks
$shop_title = ai_get_text('ecommerce.shop.title', __('Shop', 'ai-theme'));
$home_text = ai_get_text('common.home', __('Home', 'ai-theme'));

$category_name = $is_category ? $current_term->name : $shop_title;
$category_description = $is_category ? $current_term->description : ai_get_text('ecommerce.shop.description', '');

// Build breadcrumbs
$breadcrumb_items = [
    ['text' => $home_text, 'link' => ['url' => home_url('/')]],
    ['text' => $shop_title, 'link' => ['url' => get_permalink(wc_get_page_id('shop'))]],
];

// Add parent categories if nested
if ($is_category && $current_term->parent) {
    $parent_terms = get_ancestors($current_term->term_id, 'product_cat');
    $parent_terms = array_reverse($parent_terms);
    foreach ($parent_terms as $parent_id) {
        $parent = get_term($parent_id, 'product_cat');
        $breadcrumb_items[] = [
            'text' => $parent->name,
            'link' => ['url' => get_term_link($parent)]
        ];
    }
}

// Add current category (no link)
if ($is_category) {
    $breadcrumb_items[] = ['text' => $category_name, 'link' => ['url' => '']];
}
?>

<div data-theme="<?php echo esc_attr(get_option('ai_theme_style', 'modern')); ?>">

    <!-- Breadcrumbs -->
    <nav class="ai-breadcrumbs ai-breadcrumbs-simple w-full bg-gray-50 py-4" aria-label="Breadcrumb">
        <div class="ai-container">
            <ol class="flex items-center flex-wrap gap-2 text-sm">
                <?php
                $total = count($breadcrumb_items);
                foreach ($breadcrumb_items as $i => $item) :
                ?>
                    <li class="flex items-center gap-2">
                        <?php if ($i < $total - 1 && !empty($item['link']['url'])) : ?>
                            <a href="<?php echo esc_url($item['link']['url']); ?>" class="text-gray-600 hover:text-gray-900 transition-colors hover:underline"><?php echo esc_html($item['text']); ?></a>
                            <span class="ai-breadcrumbs-sep text-gray-400" aria-hidden="true">/</span>
                        <?php else : ?>
                            <span class="ai-breadcrumbs-current text-gray-900 font-medium" aria-current="page"><?php echo esc_html($item['text']); ?></span>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ol>
        </div>
    </nav>

    <!-- Category Header -->
    <div class="ai-category-header bg-white py-8 border-b border-gray-200">
        <div class="ai-container">
            <h1 class="text-3xl font-bold text-gray-900 mb-2"><?php echo esc_html($category_name); ?></h1>
            <?php if ($category_description) : ?>
            <p class="text-gray-600 text-lg max-w-3xl"><?php echo wp_kses_post($category_description); ?></p>
            <?php endif; ?>

            <?php
            // Show subcategories if any
            if ($is_category) {
                $subcategories = get_terms([
                    'taxonomy' => 'product_cat',
                    'parent' => $current_term->term_id,
                    'hide_empty' => true,
                ]);

                if (!empty($subcategories) && !is_wp_error($subcategories)) :
            ?>
            <div class="ai-subcategories flex flex-wrap gap-2 mt-4">
                <?php foreach ($subcategories as $subcat) : ?>
                <a href="<?php echo esc_url(get_term_link($subcat)); ?>" class="inline-flex items-center px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm text-gray-700 transition-colors">
                    <?php echo esc_html($subcat->name); ?>
                    <span class="ml-2 text-gray-500">(<?php echo esc_html($subcat->count); ?>)</span>
                </a>
                <?php endforeach; ?>
            </div>
            <?php
                endif;
            }
            ?>
        </div>
    </div>

    <!-- Products Grid -->
    <div class="ai-products-archive bg-gray-50 py-12">
        <div class="ai-container">

            <?php if (woocommerce_product_loop()) : ?>

            <!-- Sorting and Count -->
            <div class="ai-products-toolbar flex flex-wrap items-center justify-between gap-4 mb-8">
                <p class="text-gray-600">
                    <?php
                    $total = wc_get_loop_prop('total');
                    $showing_text = ai_get_text('ecommerce.shop.showing', '');
                    $products_text = ai_get_text('ecommerce.cart.itemsLabel', _n('product', 'products', $total, 'ai-theme'));
                    if ($showing_text) {
                        echo esc_html($showing_text) . ' ' . esc_html($total) . ' ' . esc_html($products_text);
                    } else {
                    printf(_n('Showing %d product', 'Showing %d products', $total, 'ai-theme'), $total);
                    }
                    ?>
                </p>
                <div class="ai-products-sorting">
                    <?php woocommerce_catalog_ordering(); ?>
                </div>
            </div>

            <!-- Products -->
            <div class="ai-products-grid grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                <?php
                while (have_posts()) :
                    the_post();
                    global $product;

                    $product_id = $product->get_id();
                    $product_name = $product->get_name();
                    $product_price = $product->get_price();
                    $product_regular_price = $product->get_regular_price();
                    $product_image = wp_get_attachment_url($product->get_image_id());
                    $product_link = get_permalink();
                    $is_on_sale = $product->is_on_sale();
                ?>
                <div class="ai-product-card bg-white rounded-xl shadow-sm hover:shadow-lg transition-shadow overflow-hidden group">
                    <!-- Image -->
                    <a href="<?php echo esc_url($product_link); ?>" class="block aspect-square overflow-hidden bg-gray-100 relative">
                        <?php if ($product_image) : ?>
                        <img src="<?php echo esc_url($product_image); ?>" alt="<?php echo esc_attr($product_name); ?>" class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300">
                        <?php else : ?>
                        <div class="w-full h-full flex items-center justify-center text-gray-400">
                            <i class="fas fa-image text-4xl"></i>
                        </div>
                        <?php endif; ?>

                        <?php if ($is_on_sale) : ?>
                        <span class="absolute top-3 left-3 bg-red-500 text-white text-xs font-bold px-2 py-1 rounded">
                            <?php echo esc_html(ai_get_text('ecommerce.product.sale', __('Sale', 'ai-theme'))); ?>
                        </span>
                        <?php endif; ?>
                    </a>

                    <!-- Content -->
                    <div class="p-4">
                        <a href="<?php echo esc_url($product_link); ?>" class="block">
                            <h3 class="font-semibold text-gray-900 mb-2 line-clamp-2 hover:text-primary transition-colors">
                                <?php echo esc_html($product_name); ?>
                            </h3>
                        </a>

                        <!-- Price -->
                        <div class="flex items-baseline gap-2">
                            <?php if ($is_on_sale && $product_regular_price) : ?>
                            <span class="text-lg font-bold text-gray-900"><?php echo wc_price($product->get_sale_price()); ?></span>
                            <span class="text-sm text-gray-500 line-through"><?php echo wc_price($product_regular_price); ?></span>
                            <?php else : ?>
                            <span class="text-lg font-bold text-gray-900"><?php echo wc_price($product_price); ?></span>
                            <?php endif; ?>
                        </div>

                        <!-- Add to Cart -->
                        <div class="mt-4">
                            <?php if ($product->is_purchasable() && $product->is_in_stock()) : ?>
                            <a href="<?php echo esc_url($product->add_to_cart_url()); ?>"
                               data-quantity="1"
                               data-product_id="<?php echo esc_attr($product_id); ?>"
                               class="ajax_add_to_cart add_to_cart_button w-full inline-flex items-center justify-center gap-2 bg-primary text-white py-2 px-4 rounded-lg font-medium hover:opacity-90 transition-opacity text-sm">
                                <i class="fas fa-shopping-cart"></i>
                                <?php echo esc_html(apply_filters('woocommerce_product_add_to_cart_text', $product->add_to_cart_text(), $product)); ?>
                            </a>
                            <?php elseif (!$product->is_in_stock()) : ?>
                            <span class="w-full inline-flex items-center justify-center bg-gray-200 text-gray-600 py-2 px-4 rounded-lg text-sm">
                                <?php echo esc_html(ai_get_text('ecommerce.product.outOfStock', __('Out of Stock', 'ai-theme'))); ?>
                            </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endwhile; ?>
            </div>

            <!-- Pagination -->
            <div class="ai-products-pagination mt-12">
                <?php
                $total_pages = wc_get_loop_prop('total_pages');
                if ($total_pages > 1) :
                ?>
                <nav class="flex items-center justify-center gap-2">
                    <?php
                    echo paginate_links([
                        'total' => $total_pages,
                        'current' => max(1, get_query_var('paged')),
                        'prev_text' => '<i class="fas fa-chevron-left"></i>',
                        'next_text' => '<i class="fas fa-chevron-right"></i>',
                        'type' => 'list',
                        'before_page_number' => '<span class="inline-flex items-center justify-center w-10 h-10 rounded-lg">',
                        'after_page_number' => '</span>',
                    ]);
                    ?>
                </nav>
                <?php endif; ?>
            </div>

            <?php else : ?>

            <!-- No Products -->
            <div class="ai-no-products text-center py-16">
                <div class="text-gray-400 mb-4">
                    <i class="fas fa-box-open text-6xl"></i>
                </div>
                <h2 class="text-2xl font-bold text-gray-900 mb-2"><?php echo esc_html(ai_get_text('ecommerce.shop.noProducts', __('No products found', 'ai-theme'))); ?></h2>
                <p class="text-gray-600 mb-6"><?php echo esc_html(ai_get_text('ecommerce.shop.noProductsDescription', __('Try adjusting your search or filter to find what you\'re looking for.', 'ai-theme'))); ?></p>
                <a href="<?php echo esc_url(get_permalink(wc_get_page_id('shop'))); ?>" class="inline-flex items-center gap-2 bg-primary text-white py-3 px-6 rounded-lg font-medium hover:opacity-90 transition-opacity">
                    <i class="fas fa-arrow-left"></i>
                    <?php echo esc_html(ai_get_text('common.backTo', __('Back to Shop', 'ai-theme'))); ?>
                </a>
            </div>

            <?php endif; ?>
        </div>
    </div>

</div>

<?php
get_footer();
