<?php
/**
 * Single Post Template
 *
 * Template for displaying individual blog posts with AI Theme styling.
 */

get_header();

// Get post data
$post_id = get_the_ID();
$categories = get_the_category();
$tags = get_the_tags();
$author_id = get_the_author_meta('ID');
// Priority: AI-generated author name > WP author
$author_name = ai_get_text('blogPage.authorName', '');
if (empty($author_name)) {
    $author_name = get_the_author();
}
if (empty($author_name)) {
    $author_name = ai_get_text('blogPage.author', __('Author', 'ai-theme'));
}
$author_avatar = get_avatar_url($author_id, ['size' => 48]);
// Use localized date format based on site language
$post_date = ai_get_localized_date('j F Y', get_the_time('U'));
$reading_time = ceil(str_word_count(get_the_content()) / 200); // ~200 words per minute

// Featured image
$featured_image = get_the_post_thumbnail_url($post_id, 'large');

// Related posts
$related_posts = [];
if (!empty($categories)) {
    $related_query = new WP_Query([
        'category__in' => wp_list_pluck($categories, 'term_id'),
        'post__not_in' => [$post_id],
        'posts_per_page' => 3,
        'orderby' => 'rand',
    ]);
    $related_posts = $related_query->posts;
    wp_reset_postdata();
}
?>

<div class="min-h-screen bg-gray-50" data-theme="default">
    <!-- Navbar is rendered by header.php -->

    <!-- Breadcrumbs -->
    <div class="bg-white border-b border-gray-200">
        <div class="ai-container-narrow py-4">
            <nav class="flex items-center gap-2 text-sm text-gray-500">
                <a href="<?php echo esc_url(home_url('/')); ?>" class="hover:text-primary transition-colors"><?php echo esc_html(ai_get_text('common.home', '')); ?></a>
                <span>/</span>
                <a href="<?php echo esc_url(home_url('/blog/')); ?>" class="hover:text-primary transition-colors"><?php echo esc_html(ai_get_text('navigation.labels.blog', '')); ?></a>
                <?php if (!empty($categories)) : ?>
                    <span>/</span>
                    <a href="<?php echo get_category_link($categories[0]->term_id); ?>" class="hover:text-primary transition-colors">
                        <?php echo esc_html($categories[0]->name); ?>
                    </a>
                <?php endif; ?>
                <span>/</span>
                <span class="text-gray-900 font-medium truncate max-w-xs"><?php the_title(); ?></span>
            </nav>
        </div>
    </div>

    <article class="py-12">
        <div class="ai-container-narrow">

            <!-- Article Header -->
            <header class="mb-8">
                <!-- Categories -->
                <?php if (!empty($categories)) : ?>
                    <div class="flex flex-wrap gap-2 mb-4">
                        <?php foreach ($categories as $category) : ?>
                            <a href="<?php echo get_category_link($category->term_id); ?>"
                               class="inline-block bg-primary/10 text-primary text-sm font-medium px-3 py-1 rounded-full hover:bg-primary/20 transition-colors">
                                <?php echo esc_html($category->name); ?>
                            </a>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <!-- Title -->
                <h1 class="text-3xl md:text-4xl lg:text-5xl font-bold text-gray-900 mb-6 leading-tight">
                    <?php the_title(); ?>
                </h1>

                <!-- Meta info -->
                <div class="flex flex-wrap items-center gap-6 text-gray-600">
                    <!-- Author -->
                    <div class="flex items-center gap-3">
                        <img src="<?php echo esc_url($author_avatar); ?>"
                             alt="<?php echo esc_attr($author_name); ?>"
                             class="w-10 h-10 rounded-full object-cover">
                        <div>
                            <p class="font-medium text-gray-900"><?php echo esc_html($author_name); ?></p>
                            <p class="text-sm"><?php echo esc_html($post_date); ?></p>
                        </div>
                    </div>

                    <!-- Reading time -->
                    <div class="flex items-center gap-2 text-sm">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        <span><?php echo $reading_time; ?> <?php echo esc_html(ai_get_text('blogPage.minRead', '')); ?></span>
                    </div>
                </div>
            </header>

            <!-- Featured Image -->
            <?php if ($featured_image) : ?>
                <div class="mb-10 rounded-2xl overflow-hidden shadow-lg">
                    <img src="<?php echo esc_url($featured_image); ?>"
                         alt="<?php the_title_attribute(); ?>"
                         class="w-full h-auto object-cover">
                </div>
            <?php endif; ?>

            <!-- Article Content -->
            <div class="prose prose-lg max-w-none
                        prose-headings:font-bold prose-headings:text-gray-900
                        prose-h2:text-2xl prose-h2:mt-10 prose-h2:mb-4
                        prose-h3:text-xl prose-h3:mt-8 prose-h3:mb-3
                        prose-p:text-gray-700 prose-p:leading-relaxed prose-p:mb-4
                        prose-a:text-primary prose-a:no-underline hover:prose-a:underline
                        prose-strong:text-gray-900
                        prose-ul:my-4 prose-li:text-gray-700
                        prose-blockquote:border-l-4 prose-blockquote:border-primary prose-blockquote:pl-4 prose-blockquote:italic
                        prose-img:rounded-xl prose-img:shadow-md">
                <?php
                // Get content and convert markdown-style headers if needed
                $content = get_the_content();

                // Convert markdown headers to HTML
                $content = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $content);
                $content = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $content);
                $content = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $content);

                // Convert markdown bold
                $content = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $content);

                // Convert markdown italic
                $content = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $content);

                // Convert markdown lists
                $content = preg_replace('/^- (.+)$/m', '<li>$1</li>', $content);
                $content = preg_replace('/(<li>.+<\/li>\n?)+/', '<ul>$0</ul>', $content);

                // Apply WordPress filters and output
                echo apply_filters('the_content', $content);
                ?>
            </div>

            <!-- Tags -->
            <?php if (!empty($tags)) : ?>
                <div class="mt-10 pt-6 border-t border-gray-200">
                    <div class="flex flex-wrap items-center gap-2">
                        <span class="text-gray-600 font-medium"><?php echo esc_html(ai_get_text('blogPage.tagsHeading', '')); ?></span>
                        <?php foreach ($tags as $tag) : ?>
                            <a href="<?php echo get_tag_link($tag->term_id); ?>"
                               class="inline-block bg-gray-100 text-gray-700 text-sm px-3 py-1 rounded-lg hover:bg-gray-200 transition-colors">
                                #<?php echo esc_html($tag->name); ?>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Share buttons -->
            <div class="mt-8 pt-6 border-t border-gray-200">
                <div class="flex items-center gap-4">
                    <span class="text-gray-600 font-medium"><?php echo esc_html(ai_get_text('blogPage.shareButton', '')); ?></span>
                    <a href="https://twitter.com/intent/tweet?url=<?php echo urlencode(get_permalink()); ?>&text=<?php echo urlencode(get_the_title()); ?>"
                       target="_blank" rel="noopener noreferrer"
                       class="w-10 h-10 flex items-center justify-center rounded-full bg-gray-100 text-gray-600 hover:bg-blue-500 hover:text-white transition-colors">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/>
                        </svg>
                    </a>
                    <a href="https://www.facebook.com/sharer/sharer.php?u=<?php echo urlencode(get_permalink()); ?>"
                       target="_blank" rel="noopener noreferrer"
                       class="w-10 h-10 flex items-center justify-center rounded-full bg-gray-100 text-gray-600 hover:bg-blue-600 hover:text-white transition-colors">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
                        </svg>
                    </a>
                    <a href="https://www.linkedin.com/shareArticle?mini=true&url=<?php echo urlencode(get_permalink()); ?>&title=<?php echo urlencode(get_the_title()); ?>"
                       target="_blank" rel="noopener noreferrer"
                       class="w-10 h-10 flex items-center justify-center rounded-full bg-gray-100 text-gray-600 hover:bg-blue-700 hover:text-white transition-colors">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/>
                        </svg>
                    </a>
                    <button onclick="navigator.clipboard.writeText('<?php echo get_permalink(); ?>')"
                            class="w-10 h-10 flex items-center justify-center rounded-full bg-gray-100 text-gray-600 hover:bg-gray-200 transition-colors"
                            title="Copy link">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2m0 0h2a2 2 0 012 2v3m2 4H10m0 0l3-3m-3 3l3 3"></path>
                        </svg>
                    </button>
                </div>
            </div>

        </div>
    </article>

    <!-- Related Posts -->
    <?php if (!empty($related_posts)) : ?>
        <section class="py-12 bg-white border-t border-gray-200">
            <div class="ai-container">
                <h2 class="text-2xl font-bold text-gray-900 mb-8"><?php echo esc_html(ai_get_text('blogPage.relatedArticles', '')); ?></h2>

                <div class="grid md:grid-cols-3 gap-8">
                    <?php foreach ($related_posts as $related) :
                        $related_image = get_the_post_thumbnail_url($related->ID, 'medium');
                        $related_categories = get_the_category($related->ID);
                    ?>
                        <article class="bg-gray-50 rounded-xl overflow-hidden hover:shadow-lg transition-shadow">
                            <?php if ($related_image) : ?>
                                <a href="<?php echo get_permalink($related->ID); ?>">
                                    <img src="<?php echo esc_url($related_image); ?>"
                                         alt="<?php echo esc_attr($related->post_title); ?>"
                                         class="w-full h-48 object-cover">
                                </a>
                            <?php else : ?>
                                <div class="w-full h-48 bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
                                    <svg class="w-12 h-12 text-primary/30" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10a2 2 0 012 2v1m2 13a2 2 0 01-2-2V7m2 13a2 2 0 002-2V9a2 2 0 00-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"></path>
                                    </svg>
                                </div>
                            <?php endif; ?>

                            <div class="p-5">
                                <?php if (!empty($related_categories)) : ?>
                                    <span class="text-primary text-sm font-medium">
                                        <?php echo esc_html($related_categories[0]->name); ?>
                                    </span>
                                <?php endif; ?>

                                <h3 class="font-bold text-gray-900 mt-2 line-clamp-2">
                                    <a href="<?php echo get_permalink($related->ID); ?>" class="hover:text-primary transition-colors">
                                        <?php echo esc_html($related->post_title); ?>
                                    </a>
                                </h3>

                                <p class="text-gray-600 text-sm mt-2 line-clamp-2">
                                    <?php echo wp_trim_words($related->post_excerpt ?: $related->post_content, 20); ?>
                                </p>

                                <div class="mt-4 text-sm text-gray-500">
                                    <?php echo ai_get_localized_date('j M Y', get_post_time('U', false, $related->ID)); ?>
                                </div>
                            </div>
                        </article>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
    <?php endif; ?>

    <!-- Footer is rendered by footer.php -->
</div>

<?php get_footer(); ?>
