<?php
/**
 * Stats Dark Widget
 *
 * Cyberpunk-inspired dark theme with neon accents
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Stats_Dark extends AI_Widget_Base {

    public function get_name() {
        return 'ai_stats_dark';
    }

    public function get_title() {
        return __('Stats Dark', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-number-field';
    }

    public function get_categories() {
        return array('ai-theme-stats');
    }

    public function get_keywords() {
        return array('stats', 'statistics', 'numbers', 'dark', 'neon', 'cyberpunk', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'value',
            array(
                'label'   => __('Value', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'label',
            array(
                'label'   => __('Label', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'color',
            array(
                'label'   => __('Neon Color', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'cyan',
                'options' => array(
                    'cyan'    => __('Cyan', 'ai-theme'),
                    'purple'  => __('Purple', 'ai-theme'),
                    'pink'    => __('Pink', 'ai-theme'),
                    'emerald' => __('Emerald', 'ai-theme'),
                ),
            )
        );

        $this->add_control(
            'stats',
            array(
                'label'   => __('Stats', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        $neon_colors = array(
            'cyan'    => array('glow' => 'shadow-cyan-500/50', 'text' => 'text-cyan-400', 'border' => 'border-cyan-500/30', 'via' => 'via-cyan-400'),
            'purple'  => array('glow' => 'shadow-purple-500/50', 'text' => 'text-purple-400', 'border' => 'border-purple-500/30', 'via' => 'via-purple-400'),
            'pink'    => array('glow' => 'shadow-pink-500/50', 'text' => 'text-pink-400', 'border' => 'border-pink-500/30', 'via' => 'via-pink-400'),
            'emerald' => array('glow' => 'shadow-emerald-500/50', 'text' => 'text-emerald-400', 'border' => 'border-emerald-500/30', 'via' => 'via-emerald-400'),
        );
        ?>
        <section
            class="ai-stats ai-stats-dark relative py-24 md:py-32 bg-[var(--bg-dark)] overflow-hidden"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="stats"
            data-block-variant="stats-dark"
        >
            <!-- Grid Pattern Background -->
            <div
                class="absolute inset-0 opacity-20"
                style="background-image: linear-gradient(rgba(255,255,255,0.03) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,0.03) 1px, transparent 1px); background-size: 60px 60px;"
            ></div>

            <!-- Gradient Orbs -->
            <div class="absolute top-1/2 left-1/4 w-96 h-96 bg-cyan-500/10 rounded-full blur-[100px]"></div>
            <div class="absolute top-1/3 right-1/4 w-96 h-96 bg-purple-500/10 rounded-full blur-[100px]"></div>

            <div class="relative max-w-6xl mx-auto px-6">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-16">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-4xl md:text-5xl font-bold text-[var(--text-on-dark)] mb-4 tracking-tight">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-lg text-[var(--text-on-dark)]/60 max-w-2xl mx-auto">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <div class="grid grid-cols-2 lg:grid-cols-4 gap-6">
                    <?php foreach ($settings['stats'] as $idx => $stat) :
                        $color_key = $stat['color'] ?? array_keys($neon_colors)[$idx % 4];
                        $color = $neon_colors[$color_key] ?? $neon_colors['cyan'];
                    ?>
                        <div class="group relative p-8 rounded-2xl bg-white/5 border <?php echo esc_attr($color['border']); ?> backdrop-blur-sm hover:shadow-2xl <?php echo esc_attr($color['glow']); ?> transition-all duration-500">
                            <!-- Scan Line Effect -->
                            <div class="absolute inset-0 rounded-2xl overflow-hidden opacity-0 group-hover:opacity-100 transition-opacity">
                                <div class="absolute inset-0 bg-gradient-to-b from-transparent via-white/5 to-transparent animate-pulse"></div>
                            </div>

                            <div class="relative text-center">
                                <!-- Number with Glow -->
                                <div class="text-5xl md:text-6xl font-bold <?php echo esc_attr($color['text']); ?> mb-3 tracking-tighter">
                                    <?php echo esc_html($stat['value']); ?>
                                </div>

                                <!-- Label -->
                                <div class="text-[var(--text-on-dark)]/60 text-sm uppercase tracking-wider font-medium">
                                    <?php echo esc_html($stat['label']); ?>
                                </div>

                                <!-- Bottom Accent Line -->
                                <div class="absolute -bottom-8 left-1/2 -translate-x-1/2 w-0 group-hover:w-16 h-0.5 bg-gradient-to-r from-transparent <?php echo esc_attr($color['via']); ?> to-transparent transition-all duration-500"></div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Bottom Tech Decoration -->
                <div class="mt-16 flex justify-center items-center gap-4 opacity-30">
                    <div class="h-px w-20 bg-gradient-to-r from-transparent to-cyan-500"></div>
                    <div class="w-2 h-2 rotate-45 border border-cyan-500"></div>
                    <div class="h-px w-40 bg-gradient-to-r from-cyan-500 via-purple-500 to-pink-500"></div>
                    <div class="w-2 h-2 rotate-45 border border-pink-500"></div>
                    <div class="h-px w-20 bg-gradient-to-l from-transparent to-pink-500"></div>
                </div>
            </div>
        </section>
        <?php
    }
}
