<?php
/**
 * Products Showcase Minimal Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

/**
 * Products Showcase Minimal - Clean, simple grid layout
 */
class Products_Showcase_Minimal extends AI_Widget_Base {
    public function get_name() { return 'ai_products_showcase_minimal'; }
    public function get_title() { return __('Products Showcase Minimal', 'ai-theme'); }
    public function get_icon() { return 'eicon-products'; }
    public function get_categories() { return array('ai-theme-products'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('heading', array(
            'label' => __('Heading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('subheading', array(
            'label' => __('Subheading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('count', array(
            'label' => __('Products Count', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
            'min' => 4,
            'max' => 12,
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $products = wc_get_products(array(
            'limit' => intval($settings['count']),
            'status' => 'publish',
        ));

        if (empty($products)) {
            echo '<div class="text-center py-20 bg-[var(--bg-primary)]">';
            echo '<svg class="w-16 h-16 text-[var(--text-muted)] mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/></svg>';
            echo '<p class="text-[var(--text-muted)]">' . __('No products', 'ai-theme') . '</p>';
            echo '</div>';
            return;
        }
        ?>
        <section class="w-full bg-[var(--bg-primary)] py-16 md:py-24" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="products" data-block-variant="products-showcase-minimal">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-12">
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-sm text-[var(--text-muted)] uppercase tracking-wider mb-2">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)]">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <!-- Products Grid -->
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
                    <?php foreach ($products as $product) : 
                        $image = wp_get_attachment_image_url($product->get_image_id(), 'woocommerce_thumbnail');
                        $rating = $product->get_average_rating();
                        $categories = wp_get_post_terms($product->get_id(), 'product_cat');
                        $category_name = !empty($categories) && !is_wp_error($categories) ? $categories[0]->name : '';
                        $is_on_sale = $product->is_on_sale();
                        $discount = 0;
                        if ($is_on_sale && $product->get_regular_price() && $product->get_sale_price()) {
                            $discount = round((1 - $product->get_sale_price() / $product->get_regular_price()) * 100);
                        }
                    ?>
                        <div class="group">
                            <!-- Image -->
                            <div class="relative aspect-square mb-4 overflow-hidden rounded-lg bg-[var(--bg-secondary)]">
                                <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                    <?php if ($image) : ?>
                                        <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300" />
                                    <?php endif; ?>
                                </a>

                                <?php if ($discount > 0) : ?>
                                    <div class="absolute top-3 left-3 bg-red-500 text-white px-2 py-1 rounded text-xs font-bold">
                                        -<?php echo esc_html($discount); ?>%
                                    </div>
                                <?php endif; ?>

                                <!-- Wishlist Button -->
                                <button class="absolute top-3 right-3 p-2 rounded-full bg-[var(--bg-primary)]/80 opacity-0 group-hover:opacity-100 transition-opacity hover:bg-[var(--bg-primary)]">
                                    <svg class="w-4 h-4 text-[var(--text-primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                </button>

                                <!-- Quick Add -->
                                <?php if ($product->is_in_stock()) : ?>
                                    <div class="absolute bottom-3 left-3 right-3 opacity-0 group-hover:opacity-100 transition-opacity">
                                        <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-2 rounded-lg font-medium text-sm text-center hover:opacity-90 transition-opacity flex items-center justify-center gap-2">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                            <?php _e('Add to cart', 'ai-theme'); ?>
                                        </a>
                                    </div>
                                <?php else : ?>
                                    <div class="absolute inset-0 bg-black/40 flex items-center justify-center">
                                        <span class="bg-[var(--bg-primary)] text-[var(--text-primary)] px-4 py-2 rounded font-medium text-sm">
                                            <?php _e('Out of stock', 'ai-theme'); ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                            </div>

                            <!-- Info -->
                            <div>
                                <?php if ($category_name) : ?>
                                    <p class="text-xs text-[var(--text-muted)] uppercase tracking-wide mb-1">
                                        <?php echo esc_html($category_name); ?>
                                    </p>
                                <?php endif; ?>

                                <h3 class="font-medium text-[var(--text-primary)] mb-2 line-clamp-2 group-hover:text-[var(--primary)] transition-colors">
                                    <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                        <?php echo esc_html($product->get_name()); ?>
                                    </a>
                                </h3>

                                <?php if ($rating > 0) : ?>
                                    <div class="flex items-center gap-1 mb-2">
                                        <svg class="w-4 h-4 text-yellow-400 fill-yellow-400" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                        <span class="text-sm text-[var(--text-secondary)]"><?php echo esc_html($rating); ?></span>
                                        <span class="text-sm text-[var(--text-muted)]">(<?php echo esc_html($product->get_review_count()); ?>)</span>
                                    </div>
                                <?php endif; ?>

                                <div class="flex items-baseline gap-2">
                                    <span class="text-lg font-bold text-[var(--text-primary)]">
                                        <?php echo $product->get_price_html(); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
