<?php
/**
 * Products Showcase Masonry Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

/**
 * Products Showcase Masonry - Pinterest-style dynamic layout
 */
class Products_Showcase_Masonry extends AI_Widget_Base {
    public function get_name() { return 'ai_products_showcase_masonry'; }
    public function get_title() { return __('Products Showcase Masonry', 'ai-theme'); }
    public function get_icon() { return 'eicon-products'; }
    public function get_categories() { return array('ai-theme-products'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('heading', array(
            'label' => __('Heading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('subheading', array(
            'label' => __('Subheading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('count', array(
            'label' => __('Products Count', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
            'min' => 4,
            'max' => 12,
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $products = wc_get_products(array(
            'limit' => intval($settings['count']),
            'status' => 'publish',
        ));

        if (empty($products)) {
            echo '<div class="text-center py-20"><p class="text-[var(--text-muted)]">' . __('No products found', 'ai-theme') . '</p></div>';
            return;
        }

        // Heights for masonry effect
        $heights = array('h-[380px]', 'h-[440px]', 'h-[360px]', 'h-[400px]', 'h-[460px]', 'h-[350px]', 'h-[420px]', 'h-[390px]');
        
        // Color schemes by category
        $colors = array(
            'default' => array('bg' => 'from-[var(--bg-secondary)] to-[var(--bg-primary)]', 'accent' => 'bg-[var(--text-secondary)]'),
        );
        ?>
        <section class="w-full bg-[var(--bg-primary)] py-16 md:py-24" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="products" data-block-variant="products-showcase-masonry">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-16">
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-sm font-medium text-[var(--text-muted)] tracking-wide uppercase mb-3">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-3xl md:text-4xl lg:text-5xl font-bold text-[var(--text-primary)]">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <!-- Masonry Grid -->
                <div class="columns-1 sm:columns-2 lg:columns-3 xl:columns-4 gap-4">
                    <?php foreach ($products as $idx => $product) : 
                        $image = wp_get_attachment_image_url($product->get_image_id(), 'medium');
                        $height = $heights[$idx % count($heights)];
                        $categories = wp_get_post_terms($product->get_id(), 'product_cat');
                        $category_name = !empty($categories) && !is_wp_error($categories) ? $categories[0]->name : '';
                        $is_on_sale = $product->is_on_sale();
                        $discount = 0;
                        if ($is_on_sale && $product->get_regular_price() && $product->get_sale_price()) {
                            $discount = round((1 - $product->get_sale_price() / $product->get_regular_price()) * 100);
                        }
                    ?>
                        <div class="break-inside-avoid mb-4 <?php echo esc_attr($height); ?>">
                            <a href="<?php echo esc_url($product->get_permalink()); ?>" class="group relative block h-full rounded-2xl overflow-hidden bg-[var(--bg-secondary)] p-4">
                                <!-- Image -->
                                <div class="relative h-3/5 mb-4 rounded-xl overflow-hidden bg-[var(--bg-primary)]/50">
                                    <?php if ($image) : ?>
                                        <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500" />
                                    <?php endif; ?>

                                    <?php if ($discount > 0) : ?>
                                        <div class="absolute top-3 left-3 px-2 py-1 bg-[var(--primary)] text-white text-xs font-bold rounded-full">
                                            -<?php echo esc_html($discount); ?>%
                                        </div>
                                    <?php endif; ?>

                                    <!-- Quick Actions -->
                                    <div class="absolute top-3 right-3 flex flex-col gap-2 opacity-0 group-hover:opacity-100 transition-all duration-300">
                                        <button class="p-2 rounded-full bg-white shadow-md hover:scale-110 transition-transform" onclick="event.preventDefault();">
                                            <svg class="w-4 h-4 text-[var(--text-secondary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                        </button>
                                    </div>
                                </div>

                                <!-- Content -->
                                <div class="flex flex-col h-2/5">
                                    <?php if ($category_name) : ?>
                                        <span class="inline-block w-fit px-2 py-0.5 bg-[var(--primary)] text-white text-[10px] font-medium rounded-full mb-2">
                                            <?php echo esc_html($category_name); ?>
                                        </span>
                                    <?php endif; ?>

                                    <h3 class="font-semibold text-[var(--text-primary)] mb-2 line-clamp-2 group-hover:text-[var(--primary)] transition-colors">
                                        <?php echo esc_html($product->get_name()); ?>
                                    </h3>

                                    <div class="mt-auto flex items-center justify-between">
                                        <div>
                                            <span class="text-lg font-bold text-[var(--text-primary)]">
                                                <?php echo $product->get_price_html(); ?>
                                            </span>
                                        </div>

                                        <span class="p-2 rounded-full bg-[var(--primary)] text-white opacity-0 group-hover:opacity-100 hover:scale-110 transition-all duration-300">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                        </span>
                                    </div>
                                </div>
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
