<?php
/**
 * Products Showcase Carousel Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

/**
 * Products Showcase Carousel - Horizontal scrolling carousel
 */
class Products_Showcase_Carousel extends AI_Widget_Base {
    public function get_name() { return 'ai_products_showcase_carousel'; }
    public function get_title() { return __('Products Showcase Carousel', 'ai-theme'); }
    public function get_icon() { return 'eicon-products'; }
    public function get_categories() { return array('ai-theme-products'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('heading', array(
            'label' => __('Heading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('subheading', array(
            'label' => __('Subheading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('count', array(
            'label' => __('Products Count', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
            'min' => 4,
            'max' => 20,
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $widget_id = $this->get_id();

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $products = wc_get_products(array(
            'limit' => intval($settings['count']),
            'status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC',
        ));

        if (empty($products)) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('No products found', 'ai-theme') . '</p>';
            return;
        }
        ?>
        <section class="w-full bg-[var(--bg-primary)] py-16 md:py-24" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="products" data-block-variant="products-showcase-carousel">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="flex items-end justify-between mb-10">
                        <div>
                            <?php if (!empty($settings['subheading'])) : ?>
                                <p class="text-sm text-[var(--text-muted)] uppercase tracking-wider mb-2">
                                    <?php echo esc_html($settings['subheading']); ?>
                                </p>
                            <?php endif; ?>
                            <?php if (!empty($settings['heading'])) : ?>
                                <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)]">
                                    <?php echo esc_html($settings['heading']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Navigation -->
                        <div class="hidden md:flex gap-2">
                            <button class="carousel-prev-<?php echo esc_attr($widget_id); ?> p-3 rounded-full border border-[var(--border-color)] hover:bg-[var(--bg-secondary)] transition-colors">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/></svg>
                            </button>
                            <button class="carousel-next-<?php echo esc_attr($widget_id); ?> p-3 rounded-full border border-[var(--border-color)] hover:bg-[var(--bg-secondary)] transition-colors">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/></svg>
                            </button>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Carousel -->
            <div class="relative">
                <div class="carousel-container-<?php echo esc_attr($widget_id); ?> overflow-x-auto scrollbar-hide pb-4 -mx-4 px-4 md:mx-0 md:px-0">
                    <div class="carousel-track-<?php echo esc_attr($widget_id); ?> flex gap-6 pl-4 md:pl-[max(1rem,calc((100vw-80rem)/2+1rem))]">
                        <?php foreach ($products as $product) : 
                            $image = wp_get_attachment_image_url($product->get_image_id(), 'woocommerce_thumbnail');
                            $rating = $product->get_average_rating();
                            $is_on_sale = $product->is_on_sale();
                            $discount = 0;
                            if ($is_on_sale && $product->get_regular_price() && $product->get_sale_price()) {
                                $discount = round((1 - $product->get_sale_price() / $product->get_regular_price()) * 100);
                            }
                        ?>
                            <div class="flex-shrink-0 w-72 group">
                                <div class="bg-[var(--bg-secondary)] rounded-2xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300">
                                    <!-- Image -->
                                    <div class="relative aspect-square overflow-hidden">
                                        <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                            <?php if ($image) : ?>
                                                <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500" />
                                            <?php endif; ?>
                                        </a>

                                        <?php if ($discount > 0) : ?>
                                            <div class="absolute top-3 left-3 bg-red-500 text-white px-2 py-1 rounded-full text-xs font-bold">
                                                -<?php echo esc_html($discount); ?>%
                                            </div>
                                        <?php endif; ?>

                                        <!-- Quick Actions -->
                                        <div class="absolute top-3 right-3 flex flex-col gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                            <button class="p-2 bg-[var(--bg-primary)] rounded-full shadow hover:bg-red-50 transition-colors">
                                                <svg class="w-4 h-4 text-[var(--text-primary)] hover:text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                            </button>
                                        </div>

                                        <?php if ($product->is_in_stock()) : ?>
                                            <div class="absolute bottom-3 left-3 right-3 opacity-0 group-hover:opacity-100 transition-opacity">
                                                <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-2 rounded-lg font-medium text-sm text-center hover:opacity-90 transition-opacity">
                                                    <?php _e('Add to cart', 'ai-theme'); ?>
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    </div>

                                    <!-- Info -->
                                    <div class="p-4">
                                        <h3 class="font-semibold text-[var(--text-primary)] mb-2 line-clamp-2 group-hover:text-[var(--primary)] transition-colors">
                                            <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                                <?php echo esc_html($product->get_name()); ?>
                                            </a>
                                        </h3>

                                        <?php if ($rating > 0) : ?>
                                            <div class="flex items-center gap-1 mb-2">
                                                <svg class="w-4 h-4 text-yellow-400 fill-yellow-400" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                                <span class="text-sm text-[var(--text-secondary)]"><?php echo esc_html($rating); ?></span>
                                            </div>
                                        <?php endif; ?>

                                        <div class="flex items-baseline gap-2">
                                            <span class="text-lg font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </section>

        <style>
            .scrollbar-hide::-webkit-scrollbar { display: none; }
            .scrollbar-hide { -ms-overflow-style: none; scrollbar-width: none; }
        </style>

        <script>
        (function() {
            const container = document.querySelector('.carousel-container-<?php echo esc_attr($widget_id); ?>');
            const track = document.querySelector('.carousel-track-<?php echo esc_attr($widget_id); ?>');
            const prevBtn = document.querySelector('.carousel-prev-<?php echo esc_attr($widget_id); ?>');
            const nextBtn = document.querySelector('.carousel-next-<?php echo esc_attr($widget_id); ?>');
            
            if (!container || !track) return;
            
            const cardWidth = 288 + 24; // w-72 + gap-6
            
            if (prevBtn) {
                prevBtn.addEventListener('click', () => {
                    container.scrollBy({ left: -cardWidth * 2, behavior: 'smooth' });
                });
            }
            if (nextBtn) {
                nextBtn.addEventListener('click', () => {
                    container.scrollBy({ left: cardWidth * 2, behavior: 'smooth' });
                });
            }
        })();
        </script>
        <?php
    }
}
