<?php
/**
 * Product Compact Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Product Compact Widget - For quick view or modals
 */
class Product_Compact extends AI_Widget_Base {

    public function get_name() {
        return 'ai_product_compact';
    }

    public function get_title() {
        return __('Product Compact', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-single-product';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('product_id', array(
            'label' => __('Product ID', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'description' => __('Enter WooCommerce product ID or leave empty for current product', 'ai-theme'),
        ));

        $this->add_control('add_to_cart_text', array(
            'label' => __('Add to Cart Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $product_id = !empty($settings['product_id']) ? $settings['product_id'] : get_the_ID();
        $product = wc_get_product($product_id);

        if (!$product) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('Product not found', 'ai-theme') . '</p>';
            return;
        }

        $image_id = $product->get_image_id();
        $image_url = wp_get_attachment_image_url($image_id, 'woocommerce_single');
        $gallery_ids = $product->get_gallery_image_ids();
        $all_images = array_merge(array($image_id), $gallery_ids);
        $all_images = array_filter($all_images);
        
        $is_on_sale = $product->is_on_sale();
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $discount_percent = 0;
        if ($is_on_sale && $regular_price && $sale_price) {
            $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
        }
        
        $widget_id = $this->get_id();
        ?>
        <div 
            class="bg-[var(--bg-primary)] rounded-[var(--card-radius)] overflow-hidden shadow-lg max-w-2xl mx-auto"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="product"
            data-block-variant="product-compact"
        >
            <div class="flex flex-col sm:flex-row">
                <!-- Image -->
                <div class="sm:w-2/5 relative">
                    <div class="aspect-square bg-[var(--bg-secondary)]">
                        <?php if ($image_url) : ?>
                            <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover gallery-main-<?php echo esc_attr($widget_id); ?>" />
                        <?php endif; ?>
                    </div>
                    <?php if ($discount_percent > 0) : ?>
                        <div class="absolute top-2 left-2 bg-red-500 text-white text-xs px-2 py-1 rounded-full font-bold">
                            -<?php echo esc_html($discount_percent); ?>%
                        </div>
                    <?php endif; ?>
                    <?php if (count($all_images) > 1) : ?>
                        <div class="absolute bottom-2 left-2 right-2 flex gap-1 justify-center">
                            <?php foreach (array_slice($all_images, 0, 4) as $idx => $img_id) : 
                                $large_url = wp_get_attachment_image_url($img_id, 'woocommerce_single');
                            ?>
                                <button 
                                    class="w-2 h-2 rounded-full gallery-dot-<?php echo esc_attr($widget_id); ?> <?php echo $idx === 0 ? 'bg-[var(--primary)]' : 'bg-[var(--bg-primary)]/50'; ?>"
                                    data-image="<?php echo esc_url($large_url); ?>"
                                    data-index="<?php echo esc_attr($idx); ?>"
                                ></button>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Info -->
                <div class="sm:w-3/5 p-4 sm:p-6 flex flex-col">
                    <?php 
                    $brand = '';
                    if (taxonomy_exists('pa_brand')) {
                        $brands = wp_get_post_terms($product_id, 'pa_brand');
                        if (!empty($brands) && !is_wp_error($brands)) {
                            $brand = $brands[0]->name;
                        }
                    }
                    if ($brand) : ?>
                        <p class="text-xs text-[var(--text-muted)] uppercase tracking-wide mb-1"><?php echo esc_html($brand); ?></p>
                    <?php endif; ?>

                    <h2 class="text-lg font-bold text-[var(--text-primary)] mb-2 line-clamp-2"><?php echo esc_html($product->get_name()); ?></h2>

                    <?php if ($product->get_average_rating()) : ?>
                        <div class="flex items-center gap-1 mb-3">
                            <?php for ($i = 1; $i <= 5; $i++) : ?>
                                <svg class="w-3 h-3 <?php echo $i <= round($product->get_average_rating()) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                            <?php endfor; ?>
                            <span class="text-xs text-[var(--text-muted)] ml-1">(<?php echo esc_html($product->get_review_count()); ?>)</span>
                        </div>
                    <?php endif; ?>

                    <div class="flex items-baseline gap-2 mb-3">
                        <?php if ($is_on_sale) : ?>
                            <span class="text-2xl font-bold text-[var(--text-primary)]"><?php echo wc_price($sale_price); ?></span>
                            <span class="text-sm text-[var(--text-muted)] line-through"><?php echo wc_price($regular_price); ?></span>
                        <?php else : ?>
                            <span class="text-2xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                        <?php endif; ?>
                    </div>

                    <p class="text-sm mb-4 <?php echo $product->is_in_stock() ? 'text-green-600' : 'text-red-500'; ?>">
                        <?php echo $product->is_in_stock() ? __('In Stock', 'ai-theme') : __('Out of Stock', 'ai-theme'); ?>
                    </p>

                    <div class="mt-auto space-y-3">
                        <!-- Quantity -->
                        <div class="flex items-center gap-3">
                            <span class="text-sm text-[var(--text-secondary)]">Qty:</span>
                            <div class="flex items-center border border-[var(--border-color)] rounded">
                                <button class="p-2 hover:bg-[var(--bg-secondary)] qty-minus-<?php echo esc_attr($widget_id); ?>" type="button">
                                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"/></svg>
                                </button>
                                <input type="number" class="qty-input-<?php echo esc_attr($widget_id); ?> w-8 text-center text-sm font-medium border-0 bg-transparent" value="1" min="1" />
                                <button class="p-2 hover:bg-[var(--bg-secondary)] qty-plus-<?php echo esc_attr($widget_id); ?>" type="button">
                                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                                </button>
                            </div>
                        </div>

                        <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-3 text-center rounded-[var(--btn-radius)] font-semibold text-sm hover:opacity-90 flex items-center justify-center gap-2 <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                            <?php echo esc_html($settings['add_to_cart_text']); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <script>
        (function() {
            // Gallery dots
            const mainImg = document.querySelector('.gallery-main-<?php echo esc_attr($widget_id); ?>');
            const dots = document.querySelectorAll('.gallery-dot-<?php echo esc_attr($widget_id); ?>');
            dots.forEach(dot => {
                dot.addEventListener('click', function() {
                    if (mainImg) mainImg.src = this.dataset.image;
                    dots.forEach(d => {
                        d.classList.remove('bg-[var(--primary)]');
                        d.classList.add('bg-[var(--bg-primary)]/50');
                    });
                    this.classList.remove('bg-[var(--bg-primary)]/50');
                    this.classList.add('bg-[var(--primary)]');
                });
            });

            // Quantity
            const qtyInput = document.querySelector('.qty-input-<?php echo esc_attr($widget_id); ?>');
            const qtyMinus = document.querySelector('.qty-minus-<?php echo esc_attr($widget_id); ?>');
            const qtyPlus = document.querySelector('.qty-plus-<?php echo esc_attr($widget_id); ?>');
            if (qtyMinus && qtyInput) {
                qtyMinus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    if (val > 1) qtyInput.value = val - 1;
                });
            }
            if (qtyPlus && qtyInput) {
                qtyPlus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    qtyInput.value = val + 1;
                });
            }
        })();
        </script>
        <?php
    }
}
