<?php
/**
 * Navbar Transparent Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Navbar_Transparent extends AI_Widget_Base {

    public function get_name() {
        return 'ai_navbar_transparent';
    }

    public function get_title() {
        return __('Navbar Transparent', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-nav-menu';
    }

    public function get_categories() {
        return array('ai-theme-navigation');
    }

    public function get_keywords() {
        return array('navbar', 'navigation', 'menu', 'header', 'transparent', 'gradient', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'logo_text',
            array(
                'label'   => __('Logo Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'logo_image',
            array(
                'label' => __('Logo Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('CTA Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label' => __('CTA Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'link',
            array(
                'label' => __('Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'menu_items',
            array(
                'label'   => __('Menu Items', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(
                    array('title' => 'Home', 'link' => array('url' => '/')),
                    array('title' => 'About', 'link' => array('url' => '/about')),
                    array('title' => 'Contact', 'link' => array('url' => '/contact')),
                ),
            )
        );

        $this->add_control(
            'show_cart',
            array(
                'label'        => __('Show Cart', 'ai-theme'),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __('Yes', 'ai-theme'),
                'label_off'    => __('No', 'ai-theme'),
                'return_value' => 'yes',
                'default'      => '',
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $unique_id = 'navbar-transparent-' . $this->get_id();

        // WooCommerce
        $cart_url = function_exists('wc_get_cart_url') ? wc_get_cart_url() : home_url('/cart/');
        $cart_count = 0;
        if (function_exists('WC') && WC()->cart) {
            $cart_count = WC()->cart->get_cart_contents_count();
        }
        ?>
        <nav class="ai-navbar ai-navbar-transparent relative z-50 bg-[var(--bg-primary)] border-b border-[var(--border-color)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="navbar" data-block-variant="navbar-transparent" id="<?php echo esc_attr($unique_id); ?>">
            <div class="ai-container">
                <div class="flex items-center justify-between h-16">
                    <!-- Logo + Brand -->
                    <a href="<?php echo esc_url(home_url('/')); ?>" class="flex items-center gap-2">
                        <?php
                        // Handle logo_image - can be string URL or array with url
                        $logo_url = '';
                        if (!empty($settings['logo_image'])) {
                            if (is_string($settings['logo_image'])) {
                                $logo_url = $settings['logo_image'];
                            } elseif (is_array($settings['logo_image'])) {
                                $logo_url = $settings['logo_image']['url'] ?? '';
                            }
                        }
                        ?>
                        <?php if (!empty($logo_url)) : ?>
                            <img src="<?php echo esc_url($logo_url); ?>" alt="Logo" class="h-8 object-contain">
                        <?php elseif (!empty($settings['logo_text'])) : ?>
                            <div class="w-8 h-8 bg-[var(--primary)] rounded-lg flex items-center justify-center text-[var(--text-on-primary)] font-bold">
                                <?php echo esc_html(mb_substr($settings['logo_text'], 0, 1)); ?>
                            </div>
                        <?php endif; ?>
                        <?php if (!empty($settings['logo_text'])) : ?>
                            <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['logo_text']); ?></span>
                        <?php endif; ?>
                    </a>

                    <!-- Desktop Menu -->
                    <div class="hidden md:flex items-center gap-1">
                        <?php
                        $current_url = $_SERVER['REQUEST_URI'] ?? '/';
                        $current_path = parse_url($current_url, PHP_URL_PATH) ?: '/';
                        ?>
                        <?php foreach ($settings['menu_items'] as $item) :
                            $link_url = $item['link']['url'] ?? '#';
                            if (is_array($link_url)) {
                                $link_url = $link_url['url'] ?? '#';
                            }
                            $link_path = parse_url($link_url, PHP_URL_PATH) ?: '/';
                            $is_active = ($current_path === $link_path) || ($link_path === '/' && $current_path === '/');
                        ?>
                            <a
                                href="<?php echo esc_url($link_url); ?>"
                                class="px-4 py-1.5 rounded-lg font-medium transition-all <?php echo $is_active ? 'bg-[var(--bg-secondary)] text-[var(--text-primary)]' : 'text-[var(--text-secondary)] hover:text-[var(--text-primary)] hover:bg-[var(--bg-secondary)]'; ?>"
                            >
                                <?php echo esc_html($item['title']); ?>
                            </a>
                        <?php endforeach; ?>
                    </div>

                    <!-- Desktop: Cart & CTA Button -->
                    <div class="hidden md:flex items-center gap-4">
                        <?php if ($settings['show_cart'] === 'yes') : ?>
                            <a href="<?php echo esc_url($cart_url); ?>" class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg transition-colors relative">
                                <svg class="w-5 h-5 text-[var(--text-primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                                <?php if ($cart_count > 0) : ?>
                                <span class="absolute -top-1 -right-1 bg-[var(--primary)] text-[var(--text-on-primary)] text-xs w-5 h-5 rounded-full flex items-center justify-center font-bold"><?php echo esc_html($cart_count); ?></span>
                                <?php endif; ?>
                            </a>
                        <?php endif; ?>

                        <?php if (!empty($settings['cta_text'])) : ?>
                            <a href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>" class="px-4 py-2 bg-[var(--primary)] text-[var(--text-on-primary)] font-medium rounded-lg hover:opacity-90 transition-colors">
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        <?php endif; ?>
                    </div>

                    <!-- Mobile: Cart + Hamburger -->
                    <div class="flex md:hidden items-center gap-2">
                        <?php if ($settings['show_cart'] === 'yes') : ?>
                            <a href="<?php echo esc_url($cart_url); ?>" class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg transition-colors relative">
                                <svg class="w-5 h-5 text-[var(--text-primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                                <?php if ($cart_count > 0) : ?>
                                <span class="absolute -top-1 -right-1 bg-[var(--primary)] text-[var(--text-on-primary)] text-xs w-5 h-5 rounded-full flex items-center justify-center font-bold"><?php echo esc_html($cart_count); ?></span>
                                <?php endif; ?>
                            </a>
                        <?php endif; ?>

                        <button class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg ai-mobile-menu-toggle text-[var(--text-primary)]" aria-label="Toggle menu">
                            <svg class="w-6 h-6 ai-menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                            </svg>
                            <svg class="w-6 h-6 ai-close-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Mobile Menu -->
                <div class="ai-mobile-menu md:hidden hidden border-t border-[var(--border-color)]">
                    <div class="py-4 space-y-2">
                        <?php foreach ($settings['menu_items'] as $item) :
                            $link_url = $item['link']['url'] ?? '#';
                            if (is_array($link_url)) {
                                $link_url = $link_url['url'] ?? '#';
                            }
                            $link_path = parse_url($link_url, PHP_URL_PATH) ?: '/';
                            $is_active = ($current_path === $link_path) || ($link_path === '/' && $current_path === '/');
                        ?>
                            <a
                                href="<?php echo esc_url($link_url); ?>"
                                class="block px-4 py-2 rounded-lg font-medium transition-colors <?php echo $is_active ? 'bg-[var(--bg-secondary)] text-[var(--text-primary)]' : 'text-[var(--text-secondary)] hover:bg-[var(--bg-secondary)]'; ?>"
                            >
                                <?php echo esc_html($item['title']); ?>
                            </a>
                        <?php endforeach; ?>

                        <?php if (!empty($settings['cta_text'])) : ?>
                            <a href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>" class="block mx-4 mt-4 px-4 py-2 bg-[var(--primary)] text-[var(--text-on-primary)] font-medium rounded-lg text-center hover:opacity-90 transition-colors">
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </nav>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var navbar = document.getElementById('<?php echo esc_js($unique_id); ?>');
            if (navbar) {
                var toggle = navbar.querySelector('.ai-mobile-menu-toggle');
                var menu = navbar.querySelector('.ai-mobile-menu');
                var menuIcon = navbar.querySelector('.ai-menu-icon');
                var closeIcon = navbar.querySelector('.ai-close-icon');

                if (toggle && menu) {
                    toggle.addEventListener('click', function() {
                        menu.classList.toggle('hidden');
                        menuIcon.classList.toggle('hidden');
                        closeIcon.classList.toggle('hidden');
                    });
                }
            }
        });
        </script>
        <?php
    }
}
