<?php
/**
 * Hero Split Widget
 *
 * Perfect 50/50 two-column hero section with features list and social proof
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Hero_Split extends AI_Widget_Base {

    public function get_name() {
        return 'ai_hero_split';
    }

    public function get_title() {
        return __('Hero Split', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-banner';
    }

    public function get_categories() {
        return array('ai-theme-hero');
    }

    public function get_keywords() {
        return array('hero', 'banner', 'split', 'two-column', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'eyebrow',
            array(
                'label'   => __('Eyebrow Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('CTA Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label' => __('CTA Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'image',
            array(
                'label' => __('Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $this->end_controls_section();

        // Features Section
        $this->start_controls_section(
            'features_section',
            array(
                'label' => __('Features', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'description',
            array(
                'label'   => __('Description', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater->add_control(
            'icon_color',
            array(
                'label'   => __('Icon Color', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'green',
                'options' => array(
                    'green'  => __('Green', 'ai-theme'),
                    'blue'   => __('Blue', 'ai-theme'),
                    'purple' => __('Purple', 'ai-theme'),
                    'red'    => __('Red', 'ai-theme'),
                    'yellow' => __('Yellow', 'ai-theme'),
                ),
            )
        );

        $this->add_control(
            'features',
            array(
                'label'   => __('Features List', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(
                    array(
                        'title'       => __('Easy to Use', 'ai-theme'),
                        'description' => __('Get started in minutes with our intuitive interface', 'ai-theme'),
                        'icon_color'  => 'green',
                    ),
                    array(
                        'title'       => __('Lightning Fast', 'ai-theme'),
                        'description' => __('Optimized performance for the best experience', 'ai-theme'),
                        'icon_color'  => 'blue',
                    ),
                    array(
                        'title'       => __('Secure & Reliable', 'ai-theme'),
                        'description' => __('Enterprise-grade security you can trust', 'ai-theme'),
                        'icon_color'  => 'purple',
                    ),
                ),
            )
        );

        $this->end_controls_section();

        // Stats Section
        $this->start_controls_section(
            'stats_section',
            array(
                'label' => __('Social Proof Stats', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'show_stats',
            array(
                'label'        => __('Show Stats', 'ai-theme'),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __('Yes', 'ai-theme'),
                'label_off'    => __('No', 'ai-theme'),
                'return_value' => 'yes',
                'default'      => 'yes',
            )
        );

        $this->add_control(
            'stat1_value',
            array(
                'label'     => __('Stat 1 Value', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => '10k+',
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->add_control(
            'stat1_label',
            array(
                'label'     => __('Stat 1 Label', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => __('Happy Users', 'ai-theme'),
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->add_control(
            'stat2_value',
            array(
                'label'     => __('Stat 2 Value', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => '4.9/5',
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->add_control(
            'stat2_label',
            array(
                'label'     => __('Stat 2 Label', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => __('Rating', 'ai-theme'),
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->add_control(
            'stat3_value',
            array(
                'label'     => __('Stat 3 Value', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => '99.9%',
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->add_control(
            'stat3_label',
            array(
                'label'     => __('Stat 3 Label', 'ai-theme'),
                'type'      => Controls_Manager::TEXT,
                'default'   => __('Uptime', 'ai-theme'),
                'condition' => array('show_stats' => 'yes'),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        $icon_colors = array(
            'green'  => 'text-green-500',
            'blue'   => 'text-[var(--primary)]',
            'purple' => 'text-[var(--secondary)]',
            'red'    => 'text-red-500',
            'yellow' => 'text-yellow-500',
        );

        $icons = array(
            'green'  => '<svg class="w-6 h-6 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>',
            'blue'   => '<svg class="w-6 h-6 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" /></svg>',
            'purple' => '<svg class="w-6 h-6 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" /></svg>',
            'red'    => '<svg class="w-6 h-6 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" /></svg>',
            'yellow' => '<svg class="w-6 h-6 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" /></svg>',
        );
        ?>
        <div data-theme="<?php echo esc_attr($theme); ?>" class="ai-theme-block">
            <section
                class="ai-hero ai-hero-split grid grid-cols-1 lg:grid-cols-2 min-h-screen"
                data-block-type="hero"
                data-block-variant="hero-split"
            >
                <!-- Left Column - Content -->
                <div class="section-light flex items-center justify-center px-6 md:px-12 lg:px-16 py-20 lg:py-32">
                    <div class="max-w-xl w-full space-y-8">
                        <?php if (!empty($settings['eyebrow'])) : ?>
                            <div class="inline-block">
                                <span class="text-sm font-semibold uppercase tracking-wider text-[var(--text-muted)]">
                                    <?php echo esc_html($settings['eyebrow']); ?>
                                </span>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($settings['heading'])) : ?>
                            <h1 class="ai-hero-heading text-4xl md:text-5xl lg:text-6xl font-bold leading-tight text-[var(--text-primary)]">
                                <?php echo esc_html($settings['heading']); ?>
                            </h1>
                        <?php endif; ?>

                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="ai-hero-subheading text-base md:text-lg lg:text-xl leading-relaxed text-[var(--text-secondary)]">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>

                        <?php if (!empty($settings['features'])) : ?>
                            <ul class="space-y-4">
                                <?php foreach ($settings['features'] as $feature) :
                                    $color = $feature['icon_color'] ?? 'green';
                                    $color_class = $icon_colors[$color] ?? 'text-green-500';
                                    $icon = $icons[$color] ?? $icons['green'];
                                ?>
                                    <li class="flex items-start gap-3">
                                        <span class="<?php echo esc_attr($color_class); ?>">
                                            <?php echo $icon; ?>
                                        </span>
                                        <div>
                                            <div class="font-semibold text-[var(--text-primary)]"><?php echo esc_html($feature['title']); ?></div>
                                            <div class="text-sm text-[var(--text-muted)]"><?php echo esc_html($feature['description']); ?></div>
                                        </div>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <?php if (!empty($settings['cta_text'])) : ?>
                            <div class="pt-4">
                                <a href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>" class="btn-primary px-8 py-4 rounded-lg font-semibold shadow-lg hover:shadow-xl transition-all duration-300 inline-block">
                                    <?php echo esc_html($settings['cta_text']); ?>
                                </a>
                            </div>
                        <?php endif; ?>

                        <?php if ($settings['show_stats'] === 'yes') : ?>
                            <div class="flex items-center gap-8 pt-8 border-t border-[var(--border-color)] text-[var(--text-muted)]">
                                <?php if (!empty($settings['stat1_value'])) : ?>
                                    <div>
                                        <div class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['stat1_value']); ?></div>
                                        <div class="text-sm"><?php echo esc_html($settings['stat1_label']); ?></div>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($settings['stat2_value'])) : ?>
                                    <div>
                                        <div class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['stat2_value']); ?></div>
                                        <div class="text-sm"><?php echo esc_html($settings['stat2_label']); ?></div>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($settings['stat3_value'])) : ?>
                                    <div>
                                        <div class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['stat3_value']); ?></div>
                                        <div class="text-sm"><?php echo esc_html($settings['stat3_label']); ?></div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Right Column - Image -->
                <div class="relative min-h-[400px] lg:min-h-screen">
                    <?php if (!empty($settings['image']['url'])) : ?>
                        <div
                            class="absolute inset-0 bg-cover bg-center"
                            style="background-image: url(<?php echo esc_url($settings['image']['url']); ?>);"
                        ></div>
                        <div class="absolute inset-0 bg-gradient-to-br from-black/20 via-transparent to-black/20"></div>
                        <div class="absolute inset-0 opacity-10" style="background-image: linear-gradient(rgba(255,255,255,0.1) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,0.1) 1px, transparent 1px); background-size: 50px 50px;"></div>
                    <?php else : ?>
                        <div class="absolute inset-0" style="background: linear-gradient(to bottom right, var(--primary), var(--secondary));"></div>
                    <?php endif; ?>
                </div>
            </section>
        </div>
        <?php
    }
}
