<?php
/**
 * Cart Sidebar Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Sidebar Widget - Slide-out panel style
 */
class Cart_Sidebar extends AI_Widget_Base {

    public function get_name() {
        return 'ai_cart_sidebar';
    }

    public function get_title() {
        return __('Cart Sidebar', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-cart';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control('title', array('label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('checkout_text', array('label' => __('Checkout Button', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('continue_text', array('label' => __('Continue Shopping', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        // Check if WooCommerce cart is initialized (may be null during REST API requests)
        if (!WC() || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        $cart_items = $cart->get_cart();
        $item_count = $cart->get_cart_contents_count();
        ?>
        <aside
            class="w-full max-w-md bg-[var(--bg-primary)] h-full flex flex-col shadow-2xl"
            data-theme="<?php echo esc_attr($theme); ?>"
            data-block-type="cart"
            data-block-variant="cart-sidebar"
        >
            <!-- Header -->
            <div class="flex items-center justify-between p-4 border-b border-[var(--border-color)]">
                <div class="flex items-center gap-2">
                    <svg class="w-5 h-5 text-[var(--primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                    <h2 class="text-lg font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['title']); ?></h2>
                    <span class="text-sm text-[var(--text-muted)]">(<?php echo esc_html($item_count); ?> <?php _e('items', 'ai-theme'); ?>)</span>
                </div>
            </div>

            <!-- Items -->
            <div class="flex-1 overflow-y-auto p-4">
                <?php if (empty($cart_items)) : ?>
                    <div class="text-center py-12">
                        <svg class="w-16 h-16 mx-auto text-[var(--text-muted)] mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                        <p class="text-[var(--text-muted)]"><?php _e('Cart is empty', 'ai-theme'); ?></p>
                    </div>
                <?php else : ?>
                    <div class="space-y-4">
                        <?php foreach ($cart_items as $cart_item_key => $cart_item) : 
                            $product = $cart_item['data'];
                            $image = wp_get_attachment_image_url($product->get_image_id(), 'thumbnail');
                        ?>
                            <div class="flex gap-3 bg-[var(--bg-secondary)] rounded-[var(--card-radius)] p-3">
                                <?php if ($image) : ?>
                                    <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-16 h-16 object-cover rounded" />
                                <?php endif; ?>
                                <div class="flex-1 min-w-0">
                                    <p class="font-medium text-[var(--text-primary)] text-sm truncate"><?php echo esc_html($product->get_name()); ?></p>
                                    <div class="flex items-center justify-between mt-2">
                                        <span class="text-sm text-[var(--text-secondary)]">x<?php echo esc_html($cart_item['quantity']); ?></span>
                                        <span class="font-semibold text-[var(--text-primary)]">
                                            <?php echo wc_price($product->get_price() * $cart_item['quantity']); ?>
                                        </span>
                                    </div>
                                </div>
                                <a href="<?php echo esc_url(wc_get_cart_remove_url($cart_item_key)); ?>" class="self-start p-1 text-[var(--text-muted)] hover:text-red-500">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                </a>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Footer -->
            <?php if (!empty($cart_items)) : ?>
                <div class="p-4 border-t border-[var(--border-color)] bg-[var(--bg-secondary)]">
                    <div class="flex justify-between mb-2 text-sm">
                        <span class="text-[var(--text-secondary)]"><?php _e('Subtotal', 'ai-theme'); ?></span>
                        <span class="text-[var(--text-primary)]"><?php echo wc_price($cart->get_subtotal()); ?></span>
                    </div>
                    <div class="flex justify-between mb-4">
                        <span class="font-semibold text-[var(--text-primary)]"><?php _e('Total', 'ai-theme'); ?></span>
                        <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo wc_price($cart->get_total('')); ?></span>
                    </div>
                    <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-3 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90">
                        <?php echo esc_html($settings['checkout_text']); ?>
                    </a>
                </div>
            <?php endif; ?>
        </aside>
        <?php
    }
}
