<?php
/**
 * AI Widget Base Class
 *
 * Base class for all AI Theme Elementor widgets
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * AI Widget Base
 *
 * Abstract base class that all AI Theme widgets extend
 */
abstract class AI_Widget_Base extends Widget_Base {

    /**
     * Get widget keywords
     *
     * @return array
     */
    public function get_keywords() {
        return array('ai', 'theme');
    }

    /**
     * Get custom help URL
     *
     * @return string
     */
    public function get_custom_help_url() {
        return '';
    }

    /**
     * Whether the widget requires inner wrapper
     *
     * @return bool
     */
    public function has_widget_inner_wrapper(): bool {
        return false;
    }

    /**
     * Get style dependencies
     *
     * @return array
     */
    public function get_style_depends() {
        return array('ai-theme-styles', 'ai-theme-utilities', 'ai-blocks-css');
    }

    /**
     * Get script dependencies
     *
     * @return array
     */
    public function get_script_depends() {
        return array();
    }

    /**
     * Add theme style control
     *
     * Adds a control for selecting the theme style (modern, minimal, bold, etc.)
     */
    protected function add_theme_style_control() {
        $this->add_control(
            'theme_style',
            array(
                'label'   => __('Theme Style', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'inherit',
                'options' => array(
                    'inherit' => __('Inherit from Site', 'ai-theme'),
                    'modern'  => __('Modern', 'ai-theme'),
                    'minimal' => __('Minimal', 'ai-theme'),
                    'bold'    => __('Bold', 'ai-theme'),
                    'elegant' => __('Elegant', 'ai-theme'),
                ),
            )
        );
    }

    /**
     * Temporary settings storage for render_content_with_settings
     *
     * @var array|null
     */
    protected $injected_settings = null;

    /**
     * Get settings for display - override to use injected settings when available
     *
     * @return array
     */
    public function get_settings_for_display($setting_key = null) {
        // If we have injected settings, use them
        if ($this->injected_settings !== null) {
            if ($setting_key) {
                return $this->injected_settings[$setting_key] ?? null;
            }
            return $this->injected_settings;
        }
        
        // Otherwise use parent method (Elementor's method)
        return parent::get_settings_for_display($setting_key);
    }

    /**
     * Render widget content with external settings
     *
     * This method allows rendering the widget outside of Elementor context
     * by passing settings directly instead of getting them from Elementor controls.
     *
     * @param array $settings Widget settings array
     */
    public function render_content_with_settings($settings) {
        // Store settings for use in render()
        $this->injected_settings = $settings;
        
        // Call the widget's render method
        $this->render();
        
        // Clear injected settings
        $this->injected_settings = null;
    }
}
