<?php
/**
 * AI Generated Site Theme Functions
 *
 * @package AI_Theme
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Allow Application Passwords over HTTP (for Docker/dev environments)
 * WordPress requires HTTPS by default for security reasons
 */
add_filter('wp_is_application_passwords_available_for_user', '__return_true');
add_filter('wp_is_application_passwords_available', '__return_true');

define('AI_THEME_VERSION', '1.0.0');
define('AI_THEME_DIR', get_template_directory());
define('AI_THEME_URI', get_template_directory_uri());

/**
 * Get localized date based on site language
 * Uses siteTexts language detection to format month names
 */
function ai_get_localized_date($format = 'j F Y', $timestamp = null) {
    if ($timestamp === null) {
        $timestamp = current_time('timestamp');
    }
    
    // Get site language from siteTexts or detect from common.home text
    $site_texts = get_option('ai_site_texts', array());
    $home_text = isset($site_texts['common']['home']) ? $site_texts['common']['home'] : '';
    
    // Detect language based on common.home translation
    $lang = 'en'; // default
    if ($home_text) {
        if (in_array($home_text, ['Главная', 'Головна'])) $lang = 'ru';
        elseif ($home_text === 'Startseite') $lang = 'de';
        elseif ($home_text === 'Accueil') $lang = 'fr';
        elseif ($home_text === 'Inicio') $lang = 'es';
        elseif ($home_text === 'Home') $lang = 'en';
    }
    
    // Month translations
    $months = array(
        'de' => array(
            'January' => 'Januar', 'February' => 'Februar', 'March' => 'März', 
            'April' => 'April', 'May' => 'Mai', 'June' => 'Juni',
            'July' => 'Juli', 'August' => 'August', 'September' => 'September',
            'October' => 'Oktober', 'November' => 'November', 'December' => 'Dezember',
            'Jan' => 'Jan', 'Feb' => 'Feb', 'Mar' => 'Mär', 'Apr' => 'Apr',
            'May' => 'Mai', 'Jun' => 'Jun', 'Jul' => 'Jul', 'Aug' => 'Aug',
            'Sep' => 'Sep', 'Oct' => 'Okt', 'Nov' => 'Nov', 'Dec' => 'Dez'
        ),
        'ru' => array(
            'January' => 'января', 'February' => 'февраля', 'March' => 'марта',
            'April' => 'апреля', 'May' => 'мая', 'June' => 'июня',
            'July' => 'июля', 'August' => 'августа', 'September' => 'сентября',
            'October' => 'октября', 'November' => 'ноября', 'December' => 'декабря',
            'Jan' => 'янв', 'Feb' => 'фев', 'Mar' => 'мар', 'Apr' => 'апр',
            'May' => 'май', 'Jun' => 'июн', 'Jul' => 'июл', 'Aug' => 'авг',
            'Sep' => 'сен', 'Oct' => 'окт', 'Nov' => 'ноя', 'Dec' => 'дек'
        ),
        'fr' => array(
            'January' => 'janvier', 'February' => 'février', 'March' => 'mars',
            'April' => 'avril', 'May' => 'mai', 'June' => 'juin',
            'July' => 'juillet', 'August' => 'août', 'September' => 'septembre',
            'October' => 'octobre', 'November' => 'novembre', 'December' => 'décembre',
            'Jan' => 'janv', 'Feb' => 'févr', 'Mar' => 'mars', 'Apr' => 'avr',
            'May' => 'mai', 'Jun' => 'juin', 'Jul' => 'juil', 'Aug' => 'août',
            'Sep' => 'sept', 'Oct' => 'oct', 'Nov' => 'nov', 'Dec' => 'déc'
        ),
        'es' => array(
            'January' => 'enero', 'February' => 'febrero', 'March' => 'marzo',
            'April' => 'abril', 'May' => 'mayo', 'June' => 'junio',
            'July' => 'julio', 'August' => 'agosto', 'September' => 'septiembre',
            'October' => 'octubre', 'November' => 'noviembre', 'December' => 'diciembre',
            'Jan' => 'ene', 'Feb' => 'feb', 'Mar' => 'mar', 'Apr' => 'abr',
            'May' => 'may', 'Jun' => 'jun', 'Jul' => 'jul', 'Aug' => 'ago',
            'Sep' => 'sep', 'Oct' => 'oct', 'Nov' => 'nov', 'Dec' => 'dic'
        )
    );
    
    // Format date in English first
    $date_string = date($format, $timestamp);
    
    // If we have translations for this language, apply them
    if (isset($months[$lang])) {
        foreach ($months[$lang] as $en => $translated) {
            $date_string = str_replace($en, $translated, $date_string);
        }
    }
    
    return $date_string;
}

/**
 * Theme setup
 */
function ai_theme_setup() {
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('custom-logo');
    add_theme_support('html5', array(
        'search-form', 'comment-form', 'comment-list', 'gallery', 'caption', 'style', 'script',
    ));
    add_theme_support('customize-selective-refresh-widgets');
    add_theme_support('wp-block-styles');
    add_theme_support('responsive-embeds');
    add_theme_support('align-wide');
    add_theme_support('elementor');

    // WooCommerce support
    add_theme_support('woocommerce');
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider');

    register_nav_menus(array(
        'primary' => __('Primary Menu', 'ai-theme'),
        'footer'  => __('Footer Menu', 'ai-theme'),
        'mobile'  => __('Mobile Menu', 'ai-theme'),
    ));
}
add_action('after_setup_theme', 'ai_theme_setup');

/**
 * Customize WooCommerce cart page
 * Remove default cart_totals from collaterals since we have our own in the template
 */
function ai_theme_woocommerce_cart_customizations() {
    // Remove default cart totals block (we have our own styled version in cart.php)
    remove_action('woocommerce_cart_collaterals', 'woocommerce_cart_totals', 10);
}
add_action('wp', 'ai_theme_woocommerce_cart_customizations');

/**
 * Customize WooCommerce My Account menu
 * Remove Downloads endpoint (not needed for physical product stores)
 * Apply AI-generated localization for menu labels
 */
function ai_theme_customize_account_menu($items) {
    // Remove Downloads - not needed for physical products
    unset($items['downloads']);

    // Apply AI-generated labels from siteTexts
    $menu_labels = array(
        'dashboard' => ai_get_text('profile.dashboard', ''),
        'orders' => ai_get_text('profile.orders', ''),
        'edit-address' => ai_get_text('profile.addresses', ''),
        'edit-account' => ai_get_text('profile.accountDetails', ''),
        'customer-logout' => ai_get_text('profile.logOut', ''),
    );

    foreach ($menu_labels as $key => $label) {
        if (!empty($label) && isset($items[$key])) {
            $items[$key] = $label;
        }
    }

    return $items;
}
add_filter('woocommerce_account_menu_items', 'ai_theme_customize_account_menu');

/**
 * Get hardcoded translation fallbacks by locale
 * Used when AI texts and uiLabels are not available
 */
function ai_theme_get_locale_fallbacks() {
    // No locale-specific fallbacks - rely on AI siteTexts and English defaults
    return array();
}

/**
 * Get localized text with English fallback
 * Priority: AI siteTexts > uiLabels > English default
 */
function ai_theme_get_localized_text($key, $default = '') {
    // 1. Try AI-generated siteTexts first (handles both camelCase and lowercase keys)
    $ai_text = ai_get_text($key, '');
    if (!empty($ai_text)) {
        return $ai_text;
    }
    
    // 2. Try uiLabels (system translations) - convert key format
    $label_key = str_replace('ecommerce.', '', $key);
    $label_key = str_replace('.', '_', $label_key);
    $label = ai_get_label($label_key, '');
    if (!empty($label)) {
        return $label;
    }
    
    // 3. Return English default
    return $default;
}

/**
 * Get localized "Add to cart" text (wrapper for backward compatibility)
 */
function ai_theme_get_add_to_cart_text($default = '') {
    return ai_theme_get_localized_text('ecommerce.product.addToCart', $default);
}

/**
 * Customize WooCommerce "Add to cart" button text
 */
function ai_theme_add_to_cart_text($text, $product) {
    $localized = ai_theme_get_add_to_cart_text('');
    return !empty($localized) ? $localized : $text;
}
add_filter('woocommerce_product_add_to_cart_text', 'ai_theme_add_to_cart_text', 999, 2);
add_filter('woocommerce_product_single_add_to_cart_text', 'ai_theme_add_to_cart_text', 999, 2);

/**
 * Customize WooCommerce loop add to cart link HTML
 * This handles the button text in product loops/archives
 */
function ai_theme_loop_add_to_cart_link($html, $product, $args) {
    $localized = ai_theme_get_add_to_cart_text('');
    
    if (!empty($localized)) {
        // Replace the button text in the HTML
        $html = preg_replace('/(<a[^>]*>)([^<]*)(<\/a>)/i', '$1' . esc_html($localized) . '$3', $html);
    }
    
    return $html;
}
add_filter('woocommerce_loop_add_to_cart_link', 'ai_theme_loop_add_to_cart_link', 999, 3);

/**
 * Customize WooCommerce Sale badge text
 */
function ai_theme_sale_flash($html, $post, $product) {
    // Use unified localization with fallback
    $ai_text = ai_theme_get_localized_text('ecommerce.product.sale', '');
    if (!empty($ai_text)) {
        return '<span class="onsale">' . esc_html($ai_text) . '</span>';
    }
    return $html;
}
add_filter('woocommerce_sale_flash', 'ai_theme_sale_flash', 10, 3);

/**
 * Customize WooCommerce catalog sorting options using AI-generated siteTexts
 */
function ai_theme_catalog_orderby($options) {
    // Get AI-generated texts for sorting options
    $sorting_labels = array(
        'menu_order' => ai_get_text('ecommerce.sorting.default', ''),
        'popularity' => ai_get_text('ecommerce.sorting.popularity', ''),
        'rating' => ai_get_text('ecommerce.sorting.rating', ''),
        'date' => ai_get_text('ecommerce.sorting.newest', ''),
        'price' => ai_get_text('ecommerce.sorting.priceLowToHigh', ''),
        'price-desc' => ai_get_text('ecommerce.sorting.priceHighToLow', ''),
    );
    
    foreach ($sorting_labels as $key => $label) {
        if (!empty($label) && isset($options[$key])) {
            $options[$key] = $label;
        }
    }
    
    return $options;
}
add_filter('woocommerce_catalog_orderby', 'ai_theme_catalog_orderby', 10, 1);
add_filter('woocommerce_default_catalog_orderby_options', 'ai_theme_catalog_orderby', 10, 1);

/**
 * Customize WooCommerce registration privacy policy text
 * This replaces "Your personal data will be used to support your experience..."
 */
function ai_theme_privacy_policy_text($text) {
    $ai_text = ai_get_text('account.privacyPolicyText', '');
    if (!empty($ai_text)) {
        // Add link to privacy policy page if configured
        $privacy_page_id = wc_privacy_policy_page_id();
        if ($privacy_page_id) {
            $privacy_link = '<a href="' . esc_url(get_permalink($privacy_page_id)) . '" class="text-primary hover:underline" target="_blank">' . ai_get_text('legal.privacy.title', __('privacy policy', 'woocommerce')) . '</a>';
            // If AI text doesn't contain link, append it
            if (strpos($ai_text, '</a>') === false && strpos($ai_text, 'href') === false) {
                $ai_text = str_replace(ai_get_text('legal.privacy.title', ''), $privacy_link, $ai_text);
            }
        }
        return $ai_text;
    }
    return $text;
}
add_filter('woocommerce_registration_privacy_policy_text', 'ai_theme_privacy_policy_text', 999, 1);
add_filter('woocommerce_get_privacy_policy_text', 'ai_theme_privacy_policy_text', 999, 1);

/**
 * Customize WooCommerce "Place order" button text on checkout page
 */
function ai_theme_order_button_text($text) {
    $localized = ai_theme_get_localized_text('ecommerce.checkout.placeOrder', '');
    if (!empty($localized)) {
        return $localized;
    }
    return $text;
}
add_filter('woocommerce_order_button_text', 'ai_theme_order_button_text', 10, 1);

/**
 * Customize WooCommerce "Proceed to checkout" button in cart
 */
function ai_theme_custom_proceed_to_checkout() {
    $checkout_url = wc_get_checkout_url();
    $button_text = ai_theme_get_localized_text('ecommerce.cart.proceedToCheckout', __('Proceed to checkout', 'woocommerce'));
    ?>
    <a href="<?php echo esc_url($checkout_url); ?>" class="checkout-button button alt wc-forward">
        <?php echo esc_html($button_text); ?>
    </a>
    <?php
}
// Replace default proceed to checkout button with localized version
remove_action('woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20);
add_action('woocommerce_proceed_to_checkout', 'ai_theme_custom_proceed_to_checkout', 20);

/**
 * Customize WooCommerce breadcrumbs
 */
function ai_theme_wc_breadcrumb_defaults($defaults) {
    $home_text = ai_get_text('common.home', '');
    if (!empty($home_text)) {
        $defaults['home'] = $home_text;
    }
    return $defaults;
}
add_filter('woocommerce_breadcrumb_defaults', 'ai_theme_wc_breadcrumb_defaults', 10, 1);

/**
 * Customize WooCommerce checkout field labels with AI-generated texts
 */
function ai_theme_checkout_fields($fields) {
    // Billing fields
    $billing_labels = [
        'billing_first_name' => ai_theme_get_localized_text('ecommerce.checkout.firstName', ''),
        'billing_last_name' => ai_theme_get_localized_text('ecommerce.checkout.lastName', ''),
        'billing_company' => ai_theme_get_localized_text('ecommerce.checkout.companyName', ''),
        'billing_country' => ai_theme_get_localized_text('ecommerce.checkout.country', ''),
        'billing_address_1' => ai_theme_get_localized_text('ecommerce.checkout.streetAddress', ''),
        'billing_address_2' => ai_theme_get_localized_text('ecommerce.checkout.apartment', ''),
        'billing_city' => ai_theme_get_localized_text('ecommerce.checkout.townCity', ''),
        'billing_state' => ai_theme_get_localized_text('ecommerce.checkout.state', ''),
        'billing_postcode' => ai_theme_get_localized_text('ecommerce.checkout.postcode', ''),
        'billing_phone' => ai_theme_get_localized_text('ecommerce.checkout.phone', ''),
        'billing_email' => ai_theme_get_localized_text('ecommerce.checkout.emailAddress', ''),
    ];
    
    foreach ($billing_labels as $field_key => $label) {
        if (!empty($label) && isset($fields['billing'][$field_key])) {
            $fields['billing'][$field_key]['label'] = $label;
        }
    }
    
    // Billing placeholders
    $billing_placeholders = [
        'billing_address_1' => ai_theme_get_localized_text('ecommerce.checkout.streetAddress', ''),
        'billing_address_2' => ai_theme_get_localized_text('ecommerce.checkout.apartment', ''),
    ];
    
    foreach ($billing_placeholders as $field_key => $placeholder) {
        if (!empty($placeholder) && isset($fields['billing'][$field_key])) {
            $fields['billing'][$field_key]['placeholder'] = $placeholder;
        }
    }
    
    // Shipping fields (same labels)
    $shipping_labels = [
        'shipping_first_name' => ai_theme_get_localized_text('ecommerce.checkout.firstName', ''),
        'shipping_last_name' => ai_theme_get_localized_text('ecommerce.checkout.lastName', ''),
        'shipping_company' => ai_theme_get_localized_text('ecommerce.checkout.companyName', ''),
        'shipping_country' => ai_theme_get_localized_text('ecommerce.checkout.country', ''),
        'shipping_address_1' => ai_theme_get_localized_text('ecommerce.checkout.streetAddress', ''),
        'shipping_address_2' => ai_theme_get_localized_text('ecommerce.checkout.apartment', ''),
        'shipping_city' => ai_theme_get_localized_text('ecommerce.checkout.townCity', ''),
        'shipping_state' => ai_theme_get_localized_text('ecommerce.checkout.state', ''),
        'shipping_postcode' => ai_theme_get_localized_text('ecommerce.checkout.postcode', ''),
    ];
    
    foreach ($shipping_labels as $field_key => $label) {
        if (!empty($label) && isset($fields['shipping'][$field_key])) {
            $fields['shipping'][$field_key]['label'] = $label;
        }
    }
    
    // Order notes
    $order_notes_label = ai_get_text('ecommerce.checkout.orderNotes', '');
    if (!empty($order_notes_label) && isset($fields['order']['order_comments'])) {
        $fields['order']['order_comments']['label'] = $order_notes_label;
    }
    
    return $fields;
}
add_filter('woocommerce_checkout_fields', 'ai_theme_checkout_fields', 10, 1);

/**
 * Customize WooCommerce order review table headers
 */
function ai_theme_order_review_heading($text) {
    // Product column header
    $product_text = ai_theme_get_localized_text('ecommerce.checkout.product', '');
    if (!empty($product_text)) {
        $text = str_replace('Product', $product_text, $text);
    }
    return $text;
}

/**
 * Customize checkout order subtotals labels
 */
function ai_theme_checkout_subtotals($name, $cart_subtotal) {
    $subtotal_text = ai_theme_get_localized_text('ecommerce.cart.subtotal', '');
    if (!empty($subtotal_text) && strpos($name, 'Subtotal') !== false) {
        return $subtotal_text;
    }
    return $name;
}
add_filter('woocommerce_cart_totals_order_total_label', 'ai_theme_checkout_subtotals', 10, 2);

/**
 * Customize "no payment methods" message
 */
function ai_theme_no_payment_methods_message($message) {
    $ai_text = ai_theme_get_localized_text('ecommerce.checkout.noPaymentMethods', '');
    if (!empty($ai_text)) {
        return $ai_text;
    }
    return $message;
}
add_filter('woocommerce_no_available_payment_methods_message', 'ai_theme_no_payment_methods_message', 10, 1);

/**
 * Customize "Have a coupon?" message on checkout page
 */
function ai_theme_checkout_coupon_message($message) {
    $have_coupon = ai_theme_get_localized_text('ecommerce.checkout.haveCoupon', '');
    $click_here = ai_theme_get_localized_text('ecommerce.checkout.clickToEnterCode', '');
    
    if (!empty($have_coupon) && !empty($click_here)) {
        return $have_coupon . ' <a href="#" class="showcoupon">' . $click_here . '</a>';
    }
    return $message;
}
add_filter('woocommerce_checkout_coupon_message', 'ai_theme_checkout_coupon_message', 999, 1);

/**
 * Customize checkout privacy policy text
 */
function ai_theme_checkout_privacy_policy_text($text) {
    $ai_text = ai_get_text('ecommerce.checkout.privacyPolicyText', '');
    if (!empty($ai_text)) {
        // Add link to privacy policy page if configured
        $privacy_page_id = wc_privacy_policy_page_id();
        if ($privacy_page_id) {
            $privacy_link = '<a href="' . esc_url(get_permalink($privacy_page_id)) . '" class="text-primary hover:underline" target="_blank">' . ai_get_text('legal.privacy.title', __('privacy policy', 'woocommerce')) . '</a>';
            // Replace placeholder if exists
            if (strpos($ai_text, '{privacy_policy}') !== false) {
                $ai_text = str_replace('{privacy_policy}', $privacy_link, $ai_text);
            }
        }
        return $ai_text;
    }
    return $text;
}
add_filter('woocommerce_get_privacy_policy_text', 'ai_theme_checkout_privacy_policy_text', 999, 2);
add_filter('woocommerce_checkout_privacy_policy_text', 'ai_theme_checkout_privacy_policy_text', 999, 1);

/**
 * Customize WooCommerce product tab titles
 */
function ai_theme_product_tabs($tabs) {
    if (isset($tabs['description'])) {
        $text = ai_get_text('ecommerce.product.description', '');
        if (!empty($text)) {
            $tabs['description']['title'] = $text;
        }
    }
    if (isset($tabs['reviews'])) {
        $text = ai_get_text('ecommerce.product.reviews', '');
        if (!empty($text)) {
            $tabs['reviews']['title'] = $text;
        }
    }
    if (isset($tabs['additional_information'])) {
        $text = ai_get_text('ecommerce.product.specifications', '');
        if (!empty($text)) {
            $tabs['additional_information']['title'] = $text;
        }
    }
    return $tabs;
}
add_filter('woocommerce_product_tabs', 'ai_theme_product_tabs', 99, 1);

/**
 * Add custom rewrite rules for blog
 */
function ai_theme_rewrite_rules() {
    add_rewrite_rule('^blog/?$', 'index.php?ai_blog_page=1', 'top');
    add_rewrite_rule('^blog/page/([0-9]+)/?$', 'index.php?ai_blog_page=1&paged=$matches[1]', 'top');

    // Flush rewrite rules once after adding them
    if (get_option('ai_theme_rewrite_rules_flushed') !== 'v2') {
        flush_rewrite_rules();
        update_option('ai_theme_rewrite_rules_flushed', 'v2');
    }
}
add_action('init', 'ai_theme_rewrite_rules');

/**
 * Register custom query vars
 */
function ai_theme_query_vars($vars) {
    $vars[] = 'ai_blog_page';
    return $vars;
}
add_filter('query_vars', 'ai_theme_query_vars');

/**
 * Load blog template for custom blog page
 */
function ai_theme_template_include($template) {
    if (get_query_var('ai_blog_page')) {
        $new_template = locate_template('page-blog.php');
        if ($new_template) {
            return $new_template;
        }
    }
    return $template;
}
add_filter('template_include', 'ai_theme_template_include');

/**
 * Enqueue styles and scripts
 */
function ai_theme_scripts() {
    // Tailwind CSS Play CDN (for development, switch to compiled CSS for production)
    wp_enqueue_script('tailwindcss', 'https://cdn.tailwindcss.com', array(), null, false);

    // Tailwind config (inline after Tailwind loads)
    // Fallback colors matching "modern" theme in themes.css
    $primary = get_option('ai_color_primary', '#3b82f6');
    $secondary = get_option('ai_color_secondary', '#ec4899');
    $text_primary = get_option('ai_color_text_primary', '#111827');
    $text_secondary = get_option('ai_color_text_secondary', '#4b5563');
    $font = get_option('ai_font_family', 'Inter');

    $tailwind_config = "tailwind.config = {
        theme: {
            extend: {
                colors: {
                    primary: '" . esc_js($primary) . "',
                    secondary: '" . esc_js($secondary) . "',
                    'text-primary': '" . esc_js($text_primary) . "',
                    'text-secondary': '" . esc_js($text_secondary) . "',
                },
                fontFamily: {
                    sans: ['" . esc_js($font) . "', 'sans-serif'],
                },
            },
        },
    }";
    wp_add_inline_script('tailwindcss', $tailwind_config, 'after');

    // Google Fonts
    wp_enqueue_style('google-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap', array(), null);

    // FontAwesome
    wp_enqueue_style('fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css', array(), '6.5.1');

    // Theme styles (fallback for non-Tailwind components)
    wp_enqueue_style('ai-theme-utilities', AI_THEME_URI . '/assets/css/utilities.css', array(), AI_THEME_VERSION);
    wp_enqueue_style('ai-theme-themes', AI_THEME_URI . '/assets/css/themes.css', array('ai-theme-utilities'), AI_THEME_VERSION);
    wp_enqueue_style('ai-theme-blocks', AI_THEME_URI . '/assets/css/block-themes.css', array('ai-theme-themes'), AI_THEME_VERSION);
    wp_enqueue_style('ai-theme-ai-blocks', AI_THEME_URI . '/assets/css/ai-blocks.css', array('ai-theme-blocks', 'fontawesome'), AI_THEME_VERSION);
    wp_enqueue_style('ai-theme-style', get_stylesheet_uri(), array('ai-theme-ai-blocks'), AI_THEME_VERSION);
    wp_enqueue_script('ai-theme-scripts', AI_THEME_URI . '/assets/js/theme.js', array(), AI_THEME_VERSION, true);
}
add_action('wp_enqueue_scripts', 'ai_theme_scripts');

/**
 * Add data-theme wrapper
 */
function ai_theme_body_open() {
    $theme = get_option('ai_theme_style', 'modern');
    echo '<div data-theme="' . esc_attr($theme) . '">';
}
add_action('wp_body_open', 'ai_theme_body_open', 1);

function ai_theme_body_close() {
    echo '</div>';
}
add_action('wp_footer', 'ai_theme_body_close', 100);

/**
 * Custom CSS variables (legacy support)
 */
function ai_theme_custom_css() {
    // Fallback colors matching "modern" theme in themes.css
    $primary = get_option('ai_color_primary', '#3b82f6');
    $primary_on_light = get_option('ai_color_primary_on_light', '');
    $primary_on_dark = get_option('ai_color_primary_on_dark', '');
    $secondary = get_option('ai_color_secondary', '#ec4899');
    $text_primary = get_option('ai_color_text_primary', '#111827');
    $text_secondary = get_option('ai_color_text_secondary', '#4b5563');
    $font = get_option('ai_font_family', 'Inter');
    $container_width = get_option('ai_container_width', '1280px');

    // Auto-contrast: fallback to primary if not set
    $primary_on_light = !empty($primary_on_light) ? $primary_on_light : $primary;
    $primary_on_dark = !empty($primary_on_dark) ? $primary_on_dark : '#ffffff';
    ?>
    <style id="ai-theme-variables">
        :root {
            --color-primary: <?php echo esc_attr($primary); ?>;
            --color-primary-on-light: <?php echo esc_attr($primary_on_light); ?>;
            --color-primary-on-dark: <?php echo esc_attr($primary_on_dark); ?>;
            --color-secondary: <?php echo esc_attr($secondary); ?>;
            --color-text-primary: <?php echo esc_attr($text_primary); ?>;
            --color-text-secondary: <?php echo esc_attr($text_secondary); ?>;
            --font-family: '<?php echo esc_attr($font); ?>', sans-serif;
            --ai-container-width: <?php echo esc_attr($container_width); ?>;
        }
        /* AI Container - replaces hardcoded max-w-* classes */
        .ai-container {
            width: 100%;
            max-width: var(--ai-container-width);
            margin-left: auto;
            margin-right: auto;
            padding-left: 1.5rem;
            padding-right: 1.5rem;
        }
        @media (min-width: 640px) {
            .ai-container {
                padding-left: 1.5rem;
                padding-right: 1.5rem;
            }
        }
        @media (min-width: 1024px) {
            .ai-container {
                padding-left: 2rem;
                padding-right: 2rem;
            }
        }
        /* Narrow container variant (for text-heavy content like FAQ, blog posts) */
        .ai-container-narrow {
            width: 100%;
            max-width: calc(var(--ai-container-width) * 0.7);
            margin-left: auto;
            margin-right: auto;
            padding-left: 1.5rem;
            padding-right: 1.5rem;
        }
        /* Wide container variant (for galleries, portfolios) */
        .ai-container-wide {
            width: 100%;
            max-width: calc(var(--ai-container-width) * 1.125);
            margin-left: auto;
            margin-right: auto;
            padding-left: 1.5rem;
            padding-right: 1.5rem;
        }
    </style>
    <?php
}
add_action('wp_head', 'ai_theme_custom_css', 5);

/**
 * Elementor support
 */
function ai_theme_register_elementor_locations($manager) {
    $manager->register_all_core_location();
}
add_action('elementor/theme/register_locations', 'ai_theme_register_elementor_locations');

/**
 * Load Elementor widgets
 *
 * Note: Elementor fires 'elementor/loaded' during plugins_loaded, which happens
 * BEFORE the theme's functions.php is loaded. So we need to check if Elementor
 * is already loaded and initialize immediately, otherwise hook for later.
 */
function ai_theme_elementor_init() {
    // widgets-loader.php handles loading the base class and all widgets
    require_once AI_THEME_DIR . '/inc/elementor/widgets-loader.php';
}

// Check if Elementor already loaded (theme loads after plugins_loaded)
if (did_action('elementor/loaded')) {
    // Elementor already loaded, initialize immediately
    ai_theme_elementor_init();
} else {
    // Elementor not yet loaded, wait for it
    add_action('elementor/loaded', 'ai_theme_elementor_init', 0);
}

/**
 * Add Elementor widget categories
 */
function ai_theme_add_elementor_categories($elements_manager) {
    $elements_manager->add_category('ai-theme-blocks', array(
        'title' => __('AI Theme Blocks', 'ai-theme'),
        'icon'  => 'fa fa-plug',
    ));
    $elements_manager->add_category('ai-theme-hero', array(
        'title' => __('AI - Hero', 'ai-theme'),
        'icon'  => 'fa fa-flag',
    ));
    $elements_manager->add_category('ai-theme-content', array(
        'title' => __('AI - Content', 'ai-theme'),
        'icon'  => 'fa fa-file-text',
    ));
    $elements_manager->add_category('ai-theme-navigation', array(
        'title' => __('AI - Navigation', 'ai-theme'),
        'icon'  => 'fa fa-bars',
    ));
}
add_action('elementor/elements/categories_registered', 'ai_theme_add_elementor_categories');

/**
 * Theme settings
 */
function ai_theme_register_settings() {
    register_setting('ai_theme_options', 'ai_theme_style');
    register_setting('ai_theme_options', 'ai_color_primary');
    register_setting('ai_theme_options', 'ai_color_primary_on_light');
    register_setting('ai_theme_options', 'ai_color_primary_on_dark');
    register_setting('ai_theme_options', 'ai_color_secondary');
    register_setting('ai_theme_options', 'ai_color_text_primary');
    register_setting('ai_theme_options', 'ai_color_text_secondary');
    register_setting('ai_theme_options', 'ai_font_family');
}
add_action('admin_init', 'ai_theme_register_settings');

/**
 * REST API for theme settings
 */
function ai_theme_register_rest_routes() {
    register_rest_route('ai-theme/v1', '/settings', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_update_settings',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));
    register_rest_route('ai-theme/v1', '/settings', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_settings',
        'permission_callback' => '__return_true',
    ));
    register_rest_route('ai-theme/v1', '/settings/favicon', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_set_site_icon',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));
}
add_action('rest_api_init', 'ai_theme_register_rest_routes');

function ai_theme_update_settings($request) {
    $params = $request->get_json_params();

    // Theme-specific options
    $keys = array('ai_theme_style', 'ai_color_primary', 'ai_color_primary_on_light', 'ai_color_primary_on_dark', 'ai_color_secondary', 'ai_color_text_primary', 'ai_color_text_secondary', 'ai_font_family', 'ai_container_width', 'ai_site_language');
    foreach ($keys as $key) {
        if (isset($params[$key])) {
            update_option($key, sanitize_text_field($params[$key]));
        }
    }

    // WordPress core options (site name, tagline)
    if (isset($params['blogname']) && !empty($params['blogname'])) {
        update_option('blogname', sanitize_text_field($params['blogname']));
    }
    if (isset($params['blogdescription'])) {
        update_option('blogdescription', sanitize_text_field($params['blogdescription']));
    }

    return new WP_REST_Response(array('success' => true), 200);
}

function ai_theme_get_settings() {
    return new WP_REST_Response(array(
        'ai_theme_style' => get_option('ai_theme_style', 'modern'),
        'ai_color_primary' => get_option('ai_color_primary', '#3b82f6'),
        'ai_color_primary_on_light' => get_option('ai_color_primary_on_light', ''),
        'ai_color_primary_on_dark' => get_option('ai_color_primary_on_dark', ''),
        'ai_color_secondary' => get_option('ai_color_secondary', '#ec4899'),
        'ai_color_text_primary' => get_option('ai_color_text_primary', '#111827'),
        'ai_color_text_secondary' => get_option('ai_color_text_secondary', '#4b5563'),
        'ai_font_family' => get_option('ai_font_family', 'Inter'),
        'ai_container_width' => get_option('ai_container_width', '1280px'),
        'ai_site_language' => get_option('ai_site_language', 'en'),
    ), 200);
}

/**
 * Set site icon (favicon) from media ID
 */
function ai_theme_set_site_icon($request) {
    $params = $request->get_json_params();
    $media_id = isset($params['mediaId']) ? intval($params['mediaId']) : 0;

    if ($media_id > 0) {
        // Check that media exists and is an image
        if (wp_attachment_is_image($media_id)) {
            update_option('site_icon', $media_id);
            return new WP_REST_Response(array(
                'success' => true,
                'mediaId' => $media_id,
                'message' => 'Site icon set successfully'
            ), 200);
        } else {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'Invalid media ID or not an image'
            ), 400);
        }
    }

    return new WP_REST_Response(array(
        'success' => false,
        'error' => 'Media ID is required'
    ), 400);
}

/**
 * Register additional REST API routes for site management
 */
function ai_theme_register_management_routes() {
    // Menu management
    register_rest_route('ai-theme/v1', '/menu', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_create_menu',
        'permission_callback' => function() { return current_user_can('edit_theme_options'); },
    ));

    register_rest_route('ai-theme/v1', '/menu/(?P<id>\d+)/items', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_add_menu_items',
        'permission_callback' => function() { return current_user_can('edit_theme_options'); },
    ));

    // Homepage setting
    register_rest_route('ai-theme/v1', '/settings/homepage', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_set_homepage',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // Post categories
    register_rest_route('ai-theme/v1', '/categories', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_create_category',
        'permission_callback' => function() { return current_user_can('manage_categories'); },
    ));

    // Contact Form 7 creation
    register_rest_route('ai-theme/v1', '/contact-form', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_create_contact_form',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    register_rest_route('ai-theme/v1', '/contact-form', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_contact_forms',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // Theme file sync endpoint
    register_rest_route('ai-theme/v1', '/sync-files', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_sync_files',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // WooCommerce setup endpoint - ensures required pages exist
    register_rest_route('ai-theme/v1', '/woocommerce/setup', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_setup_woocommerce_pages',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // Site URL update endpoint - changes siteurl and home options
    register_rest_route('ai-theme/v1', '/settings/siteurl', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_update_siteurl',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // WooCommerce permalinks configuration
    register_rest_route('ai-theme/v1', '/settings/wc-permalinks', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_update_wc_permalinks',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // UI Labels (system texts in site language)
    register_rest_route('ai-theme/v1', '/ui-labels', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_save_ui_labels',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    register_rest_route('ai-theme/v1', '/ui-labels', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_ui_labels',
        'permission_callback' => '__return_true',
    ));

    // Site Texts (AI-generated texts for modals, forms, ecommerce, profile, etc.)
    register_rest_route('ai-theme/v1', '/site-texts', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_save_site_texts',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    register_rest_route('ai-theme/v1', '/site-texts', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_site_texts',
        'permission_callback' => '__return_true',
    ));

    // Admin password change endpoint
    register_rest_route('ai-theme/v1', '/admin/change-password', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_change_admin_password',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // WordPress language setting endpoint
    register_rest_route('ai-theme/v1', '/settings/language', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_set_language',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // WooCommerce account/registration settings endpoint
    register_rest_route('ai-theme/v1', '/settings/wc-accounts', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_update_wc_account_settings',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // WooCommerce currency setting endpoint
    register_rest_route('ai-theme/v1', '/settings/wc-currency', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_update_wc_currency',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));
}
add_action('rest_api_init', 'ai_theme_register_management_routes');

/**
 * Update WooCommerce account and registration settings
 * Enables customer registration on My Account page for shops
 */
function ai_theme_update_wc_account_settings($request) {
    if (!class_exists('WooCommerce')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'WooCommerce is not installed',
        ), 400);
    }

    $params = $request->get_json_params();

    // Available options to configure
    $wc_options = array(
        // Enable registration on My Account page
        'enable_registration' => 'woocommerce_enable_myaccount_registration',
        // Enable registration during checkout
        'enable_checkout_registration' => 'woocommerce_enable_signup_and_login_from_checkout',
        // Auto-generate username from email
        'generate_username' => 'woocommerce_registration_generate_username',
        // Auto-generate password (send via email)
        'generate_password' => 'woocommerce_registration_generate_password',
        // Allow checkout without account (guest checkout)
        'enable_guest_checkout' => 'woocommerce_enable_guest_checkout',
    );

    $updated = array();

    foreach ($wc_options as $param_key => $option_key) {
        if (isset($params[$param_key])) {
            $value = $params[$param_key] ? 'yes' : 'no';
            update_option($option_key, $value);
            $updated[$param_key] = $value;
        }
    }

    return new WP_REST_Response(array(
        'success' => true,
        'updated' => $updated,
    ), 200);
}

/**
 * Update WooCommerce currency setting
 * Sets the store currency based on site language
 */
function ai_theme_update_wc_currency($request) {
    if (!class_exists('WooCommerce')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'WooCommerce is not installed',
        ), 400);
    }

    $params = $request->get_json_params();
    $currency = isset($params['currency']) ? sanitize_text_field($params['currency']) : 'USD';

    // Validate currency code (3 uppercase letters)
    if (!preg_match('/^[A-Z]{3}$/', $currency)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Invalid currency code',
        ), 400);
    }

    // Update WooCommerce currency option
    update_option('woocommerce_currency', $currency);

    // Also update currency position and format based on currency
    $currency_formats = array(
        'EUR' => array('position' => 'right_space', 'thousand_sep' => '.', 'decimal_sep' => ','),
        'RUB' => array('position' => 'right_space', 'thousand_sep' => ' ', 'decimal_sep' => ','),
        'UAH' => array('position' => 'right_space', 'thousand_sep' => ' ', 'decimal_sep' => ','),
        'GBP' => array('position' => 'left', 'thousand_sep' => ',', 'decimal_sep' => '.'),
        'USD' => array('position' => 'left', 'thousand_sep' => ',', 'decimal_sep' => '.'),
        'JPY' => array('position' => 'left', 'thousand_sep' => ',', 'decimal_sep' => '.'),
        'CNY' => array('position' => 'left', 'thousand_sep' => ',', 'decimal_sep' => '.'),
        'CHF' => array('position' => 'left_space', 'thousand_sep' => "'", 'decimal_sep' => '.'),
        'PLN' => array('position' => 'right_space', 'thousand_sep' => ' ', 'decimal_sep' => ','),
        'CZK' => array('position' => 'right_space', 'thousand_sep' => ' ', 'decimal_sep' => ','),
    );

    if (isset($currency_formats[$currency])) {
        $format = $currency_formats[$currency];
        update_option('woocommerce_currency_pos', $format['position']);
        update_option('woocommerce_price_thousand_sep', $format['thousand_sep']);
        update_option('woocommerce_price_decimal_sep', $format['decimal_sep']);
    }

    return new WP_REST_Response(array(
        'success' => true,
        'currency' => $currency,
    ), 200);
}

/**
 * Setup WooCommerce required pages (Shop, Cart, Checkout, My Account)
 * Creates pages if they don't exist and sets WooCommerce options
 */
function ai_theme_setup_woocommerce_pages($request) {
    if (!class_exists('WooCommerce')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'WooCommerce is not installed',
        ), 400);
    }

    $created_pages = array();
    $existing_pages = array();

    // Check for AI-generated page titles from request
    $params = $request->get_json_params();
    $ai_titles = isset($params['pageTitles']) ? $params['pageTitles'] : null;

    // If AI titles provided, use them; otherwise fallback to language-based defaults
    if ($ai_titles && !empty($ai_titles)) {
        $titles = array(
            'shop' => isset($ai_titles['shop']) ? sanitize_text_field($ai_titles['shop']) : 'Shop',
            'cart' => isset($ai_titles['cart']) ? sanitize_text_field($ai_titles['cart']) : 'Cart',
            'checkout' => isset($ai_titles['checkout']) ? sanitize_text_field($ai_titles['checkout']) : 'Checkout',
            'myaccount' => isset($ai_titles['myaccount']) ? sanitize_text_field($ai_titles['myaccount']) : 'My Account',
        );
    } else {
        // Fallback: use language-based defaults
        $lang = get_option('ai_site_language', 'en');

        // Translations for WooCommerce page titles
        $page_titles = array(
            'ru' => array(
                'shop' => 'Магазин',
                'cart' => 'Корзина',
                'checkout' => 'Оформление заказа',
                'myaccount' => 'Личный кабинет',
            ),
            'en' => array(
                'shop' => 'Shop',
                'cart' => 'Cart',
                'checkout' => 'Checkout',
                'myaccount' => 'My Account',
            ),
            'de' => array(
                'shop' => 'Shop',
                'cart' => 'Warenkorb',
                'checkout' => 'Kasse',
                'myaccount' => 'Mein Konto',
            ),
            'fr' => array(
                'shop' => 'Boutique',
                'cart' => 'Panier',
                'checkout' => 'Commande',
                'myaccount' => 'Mon compte',
            ),
            'es' => array(
                'shop' => 'Tienda',
                'cart' => 'Carrito',
                'checkout' => 'Pago',
                'myaccount' => 'Mi cuenta',
            ),
            'uk' => array(
                'shop' => 'Магазин',
                'cart' => 'Кошик',
                'checkout' => 'Оформлення замовлення',
                'myaccount' => 'Особистий кабінет',
            ),
        );

        // Use English as fallback if language not found
        $titles = isset($page_titles[$lang]) ? $page_titles[$lang] : $page_titles['en'];
    }

    // Define required WooCommerce pages
    // IMPORTANT: Use shortcodes instead of Gutenberg blocks so our theme templates are used!
    // Gutenberg blocks (wp:woocommerce/cart) render their own React components,
    // while shortcodes use theme's woocommerce/cart/cart.php template.
    $wc_pages = array(
        'shop' => array(
            'option' => 'woocommerce_shop_page_id',
            'title' => $titles['shop'],
            'slug' => 'shop',
            'content' => '',
        ),
        'cart' => array(
            'option' => 'woocommerce_cart_page_id',
            'title' => $titles['cart'],
            'slug' => 'cart',
            'content' => '[woocommerce_cart]',
        ),
        'checkout' => array(
            'option' => 'woocommerce_checkout_page_id',
            'title' => $titles['checkout'],
            'slug' => 'checkout',
            'content' => '[woocommerce_checkout]',
        ),
        'myaccount' => array(
            'option' => 'woocommerce_myaccount_page_id',
            'title' => $titles['myaccount'],
            'slug' => 'my-account',
            'content' => '[woocommerce_my_account]',
        ),
    );

    foreach ($wc_pages as $key => $page_data) {
        $option_value = get_option($page_data['option']);

        // Check if page exists and is valid
        $page_exists = false;
        if ($option_value) {
            $existing_page = get_post($option_value);
            if ($existing_page && $existing_page->post_status !== 'trash') {
                $page_exists = true;
                $existing_pages[$key] = array(
                    'id' => $option_value,
                    'title' => $existing_page->post_title,
                    'url' => get_permalink($option_value),
                );
            }
        }

        // Also check if page with this slug exists
        if (!$page_exists) {
            $slug_page = get_page_by_path($page_data['slug']);
            if ($slug_page && $slug_page->post_status !== 'trash') {
                update_option($page_data['option'], $slug_page->ID);
                $existing_pages[$key] = array(
                    'id' => $slug_page->ID,
                    'title' => $slug_page->post_title,
                    'url' => get_permalink($slug_page->ID),
                );
                continue;
            }
        }

        // Create page if it doesn't exist
        if (!$page_exists) {
            $page_id = wp_insert_post(array(
                'post_title' => $page_data['title'],
                'post_name' => $page_data['slug'],
                'post_content' => $page_data['content'],
                'post_status' => 'publish',
                'post_type' => 'page',
            ));

            if ($page_id && !is_wp_error($page_id)) {
                update_option($page_data['option'], $page_id);
                $created_pages[$key] = array(
                    'id' => $page_id,
                    'title' => $page_data['title'],
                    'url' => get_permalink($page_id),
                );
            }
        }
    }

    // Disable WooCommerce "Coming Soon" mode
    update_option('woocommerce_coming_soon', 'no');

    return new WP_REST_Response(array(
        'success' => true,
        'created' => $created_pages,
        'existing' => $existing_pages,
        'message' => sprintf('Created %d pages, %d already existed', count($created_pages), count($existing_pages)),
    ), 200);
}

/**
 * Update WordPress site URL and home URL
 * This changes how WordPress generates links (permalinks, assets, etc.)
 */
function ai_theme_update_siteurl($request) {
    $params = $request->get_json_params();
    $new_url = isset($params['url']) ? esc_url_raw(rtrim($params['url'], '/')) : '';

    if (empty($new_url)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'URL is required',
        ), 400);
    }

    // Validate URL format
    if (!filter_var($new_url, FILTER_VALIDATE_URL)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Invalid URL format',
        ), 400);
    }

    $old_siteurl = get_option('siteurl');
    $old_home = get_option('home');

    // Update both siteurl and home
    update_option('siteurl', $new_url);
    update_option('home', $new_url);

    // Flush rewrite rules to ensure permalinks work
    flush_rewrite_rules();

    return new WP_REST_Response(array(
        'success' => true,
        'old_siteurl' => $old_siteurl,
        'old_home' => $old_home,
        'new_url' => $new_url,
        'message' => 'Site URL updated successfully',
    ), 200);
}

/**
 * Change admin user password
 * Used to set a unique password for each exported site
 */
function ai_theme_change_admin_password($request) {
    $params = $request->get_json_params();
    $new_password = isset($params['password']) ? $params['password'] : '';
    $username = isset($params['username']) ? $params['username'] : 'admin';

    if (empty($new_password)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Password is required',
        ), 400);
    }

    // Find the user
    $user = get_user_by('login', $username);
    if (!$user) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'User not found: ' . $username,
        ), 404);
    }

    // Change the password
    wp_set_password($new_password, $user->ID);

    return new WP_REST_Response(array(
        'success' => true,
        'username' => $username,
        'message' => 'Password changed successfully',
    ), 200);
}

/**
 * Set WordPress site language (WPLANG option)
 * Supported locales: ru_RU, en_US, de_DE, fr_FR, es_ES, uk, etc.
 */
function ai_theme_set_language($request) {
    $params = $request->get_json_params();
    $locale = isset($params['locale']) ? sanitize_text_field($params['locale']) : '';

    if (empty($locale)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Locale is required',
        ), 400);
    }

    // Get current language
    $old_locale = get_option('WPLANG', '');

    // Update WPLANG option
    update_option('WPLANG', $locale);

    // Also update site_language for newer WordPress versions
    update_option('site_language', $locale);

    // Clear translation cache
    if (function_exists('wp_cache_delete')) {
        wp_cache_delete('language_switcher_locales', 'translations');
    }

    return new WP_REST_Response(array(
        'success' => true,
        'old_locale' => $old_locale,
        'new_locale' => $locale,
        'message' => 'Language updated successfully',
    ), 200);
}

/**
 * Update WooCommerce permalink structure
 * Changes category base from 'product-category' to 'catalog'
 */
function ai_theme_update_wc_permalinks($request) {
    if (!class_exists('WooCommerce')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'WooCommerce is not installed',
        ), 400);
    }

    $params = $request->get_json_params();
    $category_base = sanitize_text_field($params['category_base'] ?? 'catalog');
    $tag_base = sanitize_text_field($params['tag_base'] ?? 'product-tag');
    $product_base = sanitize_text_field($params['product_base'] ?? 'product');

    // Get current permalinks
    $old_permalinks = get_option('woocommerce_permalinks', array());

    // Update WooCommerce permalinks
    $new_permalinks = array(
        'category_base' => $category_base,
        'tag_base' => $tag_base,
        'product_base' => $product_base,
        'use_verbose_page_rules' => isset($old_permalinks['use_verbose_page_rules']) ? $old_permalinks['use_verbose_page_rules'] : false,
    );

    update_option('woocommerce_permalinks', $new_permalinks);

    // Flush rewrite rules to apply changes
    // Delete rewrite rules option to force regeneration on next page load
    delete_option('rewrite_rules');
    flush_rewrite_rules(true);

    return new WP_REST_Response(array(
        'success' => true,
        'old_permalinks' => $old_permalinks,
        'new_permalinks' => $new_permalinks,
        'message' => 'WooCommerce permalinks updated successfully',
    ), 200);
}

/**
 * Save UI Labels (system texts) to WordPress option
 * These labels are generated by sitemap-generator in the site's language
 */
function ai_theme_save_ui_labels($request) {
    $params = $request->get_json_params();

    if (empty($params)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'No UI labels provided',
        ), 400);
    }

    // Sanitize all labels recursively
    $sanitized_labels = ai_theme_sanitize_ui_labels($params);

    // Save to WordPress option
    update_option('ai_ui_labels', $sanitized_labels);

    return new WP_REST_Response(array(
        'success' => true,
        'message' => 'UI labels saved successfully',
        'labels_count' => count($sanitized_labels),
    ), 200);
}

/**
 * Get UI Labels from WordPress option
 */
function ai_theme_get_ui_labels($request) {
    $labels = get_option('ai_ui_labels', array());

    return new WP_REST_Response(array(
        'success' => true,
        'labels' => $labels,
    ), 200);
}

/**
 * Recursively sanitize UI labels
 */
function ai_theme_sanitize_ui_labels($data) {
    if (is_string($data)) {
        return sanitize_text_field($data);
    }

    if (is_array($data)) {
        $sanitized = array();
        foreach ($data as $key => $value) {
            $sanitized[sanitize_key($key)] = ai_theme_sanitize_ui_labels($value);
        }
        return $sanitized;
    }

    return $data;
}

/**
 * Helper function to get a UI label with fallback
 * Usage: ai_get_label('cart.title', 'Cart')
 */
function ai_get_label($key, $fallback = '') {
    static $labels = null;

    if ($labels === null) {
        $labels = get_option('ai_ui_labels', array());
    }

    // Support dot notation: 'cart.title' => $labels['cart']['title']
    $keys = explode('.', $key);
    $value = $labels;

    foreach ($keys as $k) {
        if (isset($value[$k])) {
            $value = $value[$k];
        } else {
            return $fallback;
        }
    }

    return is_string($value) ? $value : $fallback;
}

/**
 * Save Site Texts (AI-generated texts for modals, forms, ecommerce, profile, etc.)
 */
function ai_theme_save_site_texts($request) {
    $params = $request->get_json_params();

    if (empty($params)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'No site texts provided',
        ), 400);
    }

    // Sanitize all texts recursively
    $sanitized_texts = ai_theme_sanitize_ui_labels($params);

    // Save to WordPress option
    update_option('ai_site_texts', $sanitized_texts, false);

    return new WP_REST_Response(array(
        'success' => true,
        'message' => 'Site texts saved successfully',
        'texts_count' => count($sanitized_texts),
    ), 200);
}

/**
 * Get Site Texts from WordPress option
 */
function ai_theme_get_site_texts($request) {
    $texts = get_option('ai_site_texts', array());

    return new WP_REST_Response(array(
        'success' => true,
        'texts' => $texts,
    ), 200);
}

/**
 * Helper to get nested array value by keys
 */
function ai_get_nested_value($array, $keys) {
    $value = $array;
    foreach ($keys as $k) {
        if (is_array($value) && isset($value[$k])) {
            $value = $value[$k];
        } else {
            return null;
        }
    }
    return $value;
}

/**
 * Helper function to get AI-generated site text with fallback
 * Usage: ai_get_text('ecommerce.shop.title', 'Product Catalog')
 *        ai_get_text('profile.myAccount', 'My Account')
 * 
 * This reads from ai_site_texts option (AI-generated)
 * Falls back to ai_ui_labels (system translations) if not found
 * 
 * Note: Keys in siteTexts are stored as lowercase (e.g., 'addtocart' not 'addToCart')
 * This function tries both camelCase and lowercase variants
 */
function ai_get_text($key, $fallback = '') {
    static $texts = null;
    static $labels = null;

    if ($texts === null) {
        $texts = get_option('ai_site_texts', array());
    }
    if ($labels === null) {
        $labels = get_option('ai_ui_labels', array());
    }

    // Support dot notation: 'ecommerce.shop.title' => $texts['ecommerce']['shop']['title']
    $keys = explode('.', $key);
    
    // Try siteTexts first with original keys
    $value = ai_get_nested_value($texts, $keys);
    if (is_string($value) && !empty($value)) {
        return $value;
    }
    
    // Try siteTexts with lowercase keys (siteTexts are stored as lowercase)
    $lowercase_keys = array_map('strtolower', $keys);
    $value = ai_get_nested_value($texts, $lowercase_keys);
    if (is_string($value) && !empty($value)) {
        return $value;
    }
    
    // Fallback to uiLabels with original keys
    $value = ai_get_nested_value($labels, $keys);
    if (is_string($value) && !empty($value)) {
        return $value;
    }
    
    // Fallback to uiLabels with lowercase keys
    $value = ai_get_nested_value($labels, $lowercase_keys);
    if (is_string($value) && !empty($value)) {
        return $value;
    }

    return $fallback;
}

/**
 * Create WordPress navigation menu
 */
function ai_theme_create_menu($request) {
    $params = $request->get_json_params();
    $menu_name = sanitize_text_field($params['name'] ?? 'Main Menu');
    $location = sanitize_text_field($params['location'] ?? 'primary');
    $items = $params['items'] ?? array();

    // Check if menu exists, delete it first
    $existing_menu = wp_get_nav_menu_object($menu_name);
    if ($existing_menu) {
        wp_delete_nav_menu($existing_menu->term_id);
    }

    // Create new menu
    $menu_id = wp_create_nav_menu($menu_name);

    if (is_wp_error($menu_id)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => $menu_id->get_error_message(),
        ), 400);
    }

    // Add menu items
    $added_items = array();
    foreach ($items as $index => $item) {
        $item_data = array(
            'menu-item-title' => sanitize_text_field($item['title'] ?? ''),
            'menu-item-url' => esc_url_raw($item['url'] ?? '#'),
            'menu-item-status' => 'publish',
            'menu-item-position' => $index + 1,
        );

        // If linking to a page by ID
        if (!empty($item['page_id'])) {
            $item_data['menu-item-object-id'] = intval($item['page_id']);
            $item_data['menu-item-object'] = 'page';
            $item_data['menu-item-type'] = 'post_type';
            unset($item_data['menu-item-url']);
        } else {
            $item_data['menu-item-type'] = 'custom';
        }

        // Handle parent items
        if (!empty($item['parent_index']) && isset($added_items[$item['parent_index']])) {
            $item_data['menu-item-parent-id'] = $added_items[$item['parent_index']];
        }

        $item_id = wp_update_nav_menu_item($menu_id, 0, $item_data);
        if (!is_wp_error($item_id)) {
            $added_items[$index] = $item_id;
        }
    }

    // Assign menu to location
    $locations = get_theme_mod('nav_menu_locations', array());
    $locations[$location] = $menu_id;
    set_theme_mod('nav_menu_locations', $locations);

    return new WP_REST_Response(array(
        'success' => true,
        'menu_id' => $menu_id,
        'location' => $location,
        'items_count' => count($added_items),
    ), 200);
}

/**
 * Add items to existing menu
 */
function ai_theme_add_menu_items($request) {
    $menu_id = intval($request->get_param('id'));
    $params = $request->get_json_params();
    $items = $params['items'] ?? array();

    if (!is_nav_menu($menu_id)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Menu not found',
        ), 404);
    }

    $added_items = array();
    foreach ($items as $index => $item) {
        $item_data = array(
            'menu-item-title' => sanitize_text_field($item['title'] ?? ''),
            'menu-item-url' => esc_url_raw($item['url'] ?? '#'),
            'menu-item-status' => 'publish',
            'menu-item-position' => $index + 1,
            'menu-item-type' => 'custom',
        );

        if (!empty($item['page_id'])) {
            $item_data['menu-item-object-id'] = intval($item['page_id']);
            $item_data['menu-item-object'] = 'page';
            $item_data['menu-item-type'] = 'post_type';
        }

        $item_id = wp_update_nav_menu_item($menu_id, 0, $item_data);
        if (!is_wp_error($item_id)) {
            $added_items[] = $item_id;
        }
    }

    return new WP_REST_Response(array(
        'success' => true,
        'items_added' => count($added_items),
    ), 200);
}

/**
 * Set page as homepage
 */
function ai_theme_set_homepage($request) {
    $params = $request->get_json_params();
    $page_id = intval($params['page_id'] ?? 0);
    $blog_page_id = intval($params['blog_page_id'] ?? 0);

    if (!$page_id) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'page_id is required',
        ), 400);
    }

    // Check if page exists
    $page = get_post($page_id);
    if (!$page || $page->post_type !== 'page') {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Page not found',
        ), 404);
    }

    // Set static front page
    update_option('show_on_front', 'page');
    update_option('page_on_front', $page_id);

    // Optionally set blog page
    if ($blog_page_id) {
        update_option('page_for_posts', $blog_page_id);
    }

    // Disable WooCommerce "Coming Soon" mode - site is ready to go live
    if (class_exists('WooCommerce')) {
        update_option('woocommerce_coming_soon', 'no');
    }

    return new WP_REST_Response(array(
        'success' => true,
        'page_on_front' => $page_id,
        'page_for_posts' => $blog_page_id,
    ), 200);
}

/**
 * Create post category
 */
function ai_theme_create_category($request) {
    $params = $request->get_json_params();
    $name = sanitize_text_field($params['name'] ?? '');
    $slug = sanitize_title($params['slug'] ?? $name);
    $description = sanitize_textarea_field($params['description'] ?? '');
    $parent = intval($params['parent'] ?? 0);

    if (empty($name)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Category name is required',
        ), 400);
    }

    // Check if category exists
    $existing = get_term_by('slug', $slug, 'category');
    if ($existing) {
        return new WP_REST_Response(array(
            'success' => true,
            'category_id' => $existing->term_id,
            'existed' => true,
        ), 200);
    }

    // Create category
    $result = wp_insert_term($name, 'category', array(
        'slug' => $slug,
        'description' => $description,
        'parent' => $parent,
    ));

    if (is_wp_error($result)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => $result->get_error_message(),
        ), 400);
    }

    return new WP_REST_Response(array(
        'success' => true,
        'category_id' => $result['term_id'],
        'existed' => false,
    ), 200);
}

/**
 * Create Contact Form 7 form
 */
function ai_theme_create_contact_form($request) {
    // Check if Contact Form 7 is active
    if (!class_exists('WPCF7_ContactForm')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Contact Form 7 plugin is not active',
        ), 400);
    }

    $params = $request->get_json_params();
    $title = sanitize_text_field($params['title'] ?? 'Contact Form');
    $fields = $params['fields'] ?? array('name', 'email', 'message');
    $recipient = sanitize_email($params['recipient'] ?? get_option('admin_email'));
    $subject = sanitize_text_field($params['subject'] ?? 'New contact form submission');

    // Build form template
    $form_template = '';
    $mail_body = "From: [your-name] <[your-email]>\n";

    foreach ($fields as $field) {
        $field_name = sanitize_key($field);
        switch ($field_name) {
            case 'name':
                $form_template .= "<label>Name\n[text* your-name]</label>\n\n";
                break;
            case 'email':
                $form_template .= "<label>Email\n[email* your-email]</label>\n\n";
                break;
            case 'phone':
                $form_template .= "<label>Phone\n[tel your-phone]</label>\n\n";
                $mail_body .= "Phone: [your-phone]\n";
                break;
            case 'subject':
                $form_template .= "<label>Subject\n[text your-subject]</label>\n\n";
                $mail_body .= "Subject: [your-subject]\n";
                break;
            case 'message':
                $form_template .= "<label>Message\n[textarea* your-message]</label>\n\n";
                $mail_body .= "\nMessage:\n[your-message]";
                break;
            case 'company':
                $form_template .= "<label>Company\n[text your-company]</label>\n\n";
                $mail_body .= "Company: [your-company]\n";
                break;
        }
    }

    $form_template .= "[submit \"Send\"]";

    // Create Contact Form 7 post
    $form_post = array(
        'post_title' => $title,
        'post_type' => 'wpcf7_contact_form',
        'post_status' => 'publish',
    );

    $form_id = wp_insert_post($form_post);

    if (is_wp_error($form_id)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => $form_id->get_error_message(),
        ), 400);
    }

    // Save form content
    update_post_meta($form_id, '_form', $form_template);

    // Save mail settings
    $mail_settings = array(
        'active' => true,
        'subject' => $subject . ' - [your-subject]',
        'sender' => '[your-name] <' . $recipient . '>',
        'recipient' => $recipient,
        'body' => $mail_body,
        'additional_headers' => 'Reply-To: [your-email]',
        'attachments' => '',
        'use_html' => false,
    );
    update_post_meta($form_id, '_mail', $mail_settings);

    // Save messages
    $messages = array(
        'mail_sent_ok' => 'Thank you for your message. It has been sent.',
        'mail_sent_ng' => 'There was an error trying to send your message. Please try again later.',
        'validation_error' => 'One or more fields have an error. Please check and try again.',
        'spam' => 'There was an error trying to send your message. Please try again later.',
        'accept_terms' => 'You must accept the terms and conditions before sending your message.',
        'invalid_required' => 'This field is required.',
        'invalid_too_long' => 'This field is too long.',
        'invalid_too_short' => 'This field is too short.',
        'invalid_email' => 'Please enter a valid email address.',
        'invalid_tel' => 'Please enter a valid phone number.',
    );
    update_post_meta($form_id, '_messages', $messages);

    // Get shortcode
    $shortcode = '[contact-form-7 id="' . $form_id . '" title="' . esc_attr($title) . '"]';

    return new WP_REST_Response(array(
        'success' => true,
        'form_id' => $form_id,
        'shortcode' => $shortcode,
        'title' => $title,
    ), 200);
}

/**
 * Get Contact Form 7 forms
 */
function ai_theme_get_contact_forms() {
    if (!class_exists('WPCF7_ContactForm')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Contact Form 7 plugin is not active',
            'forms' => array(),
        ), 200);
    }

    $forms = get_posts(array(
        'post_type' => 'wpcf7_contact_form',
        'posts_per_page' => -1,
        'post_status' => 'publish',
    ));

    $result = array();
    foreach ($forms as $form) {
        $result[] = array(
            'id' => $form->ID,
            'title' => $form->post_title,
            'shortcode' => '[contact-form-7 id="' . $form->ID . '" title="' . esc_attr($form->post_title) . '"]',
        );
    }

    return new WP_REST_Response(array(
        'success' => true,
        'forms' => $result,
    ), 200);
}

/**
 * Sync theme files from API
 * Allows updating CSS and other theme files on re-export
 */
function ai_theme_sync_files($request) {
    $params = $request->get_json_params();
    $files = $params['files'] ?? array();

    if (empty($files)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'No files provided',
        ), 400);
    }

    $results = array();
    $theme_dir = get_template_directory();

    // Allowed paths within theme directory (security)
    $allowed_paths = array(
        'assets/css/',
        'assets/js/',
    );

    foreach ($files as $file) {
        $path = sanitize_text_field($file['path'] ?? '');
        $content = $file['content'] ?? '';

        if (empty($path) || empty($content)) {
            $results[] = array(
                'path' => $path,
                'success' => false,
                'error' => 'Missing path or content',
            );
            continue;
        }

        // Security check: ensure path is within allowed directories
        $is_allowed = false;
        foreach ($allowed_paths as $allowed) {
            if (strpos($path, $allowed) === 0) {
                $is_allowed = true;
                break;
            }
        }

        if (!$is_allowed) {
            $results[] = array(
                'path' => $path,
                'success' => false,
                'error' => 'Path not allowed',
            );
            continue;
        }

        // Prevent directory traversal
        if (strpos($path, '..') !== false) {
            $results[] = array(
                'path' => $path,
                'success' => false,
                'error' => 'Invalid path',
            );
            continue;
        }

        $full_path = $theme_dir . '/' . $path;

        // Ensure directory exists
        $dir = dirname($full_path);
        if (!file_exists($dir)) {
            wp_mkdir_p($dir);
        }

        // Write file
        $written = file_put_contents($full_path, $content);

        if ($written !== false) {
            $results[] = array(
                'path' => $path,
                'success' => true,
                'bytes' => $written,
            );
        } else {
            $results[] = array(
                'path' => $path,
                'success' => false,
                'error' => 'Failed to write file',
            );
        }
    }

    $all_success = !empty($results) && !array_filter($results, function($r) {
        return !$r['success'];
    });

    return new WP_REST_Response(array(
        'success' => $all_success,
        'results' => $results,
    ), $all_success ? 200 : 207);
}

/**
 * REST API endpoint for saving product page template blocks
 * Used to inject Elementor widgets into WooCommerce product pages
 */
function ai_theme_register_product_template_routes() {
    // Save product page template blocks
    register_rest_route('ai-theme/v1', '/product-template', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_save_product_template',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // Get product page template blocks
    register_rest_route('ai-theme/v1', '/product-template', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_product_template',
        'permission_callback' => '__return_true',
    ));

    // Save global navbar and footer blocks
    register_rest_route('ai-theme/v1', '/global-blocks', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_save_global_blocks',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ));

    // Get global navbar and footer blocks
    register_rest_route('ai-theme/v1', '/global-blocks', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_get_global_blocks',
        'permission_callback' => '__return_true',
    ));
}
add_action('rest_api_init', 'ai_theme_register_product_template_routes');

/**
 * Save product page template blocks to WordPress option
 */
function ai_theme_save_product_template($request) {
    $params = $request->get_json_params();
    $blocks = $params['blocks'] ?? array();

    if (empty($blocks)) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'No blocks provided',
        ), 400);
    }

    // Sanitize and save blocks
    // We store the entire Elementor-ready structure
    update_option('ai_product_page_blocks', $blocks, false);

    return new WP_REST_Response(array(
        'success' => true,
        'blocks_count' => count($blocks),
    ), 200);
}

/**
 * Get product page template blocks
 */
function ai_theme_get_product_template() {
    $blocks = get_option('ai_product_page_blocks', array());

    return new WP_REST_Response(array(
        'success' => true,
        'blocks' => $blocks,
    ), 200);
}

/**
 * Save global navbar and footer blocks to WordPress options
 */
function ai_theme_save_global_blocks($request) {
    $params = $request->get_json_params();
    $navbar = $params['navbar'] ?? null;
    $footer = $params['footer'] ?? null;

    // Debug logging
    if ($navbar) {
        error_log('[AI Theme] Saving navbar: variant=' . ($navbar['variant'] ?? 'none') . ', id=' . ($navbar['id'] ?? 'none'));
        update_option('ai_global_navbar', $navbar, false);
    }
    if ($footer) {
        error_log('[AI Theme] Saving footer: variant=' . ($footer['variant'] ?? 'none') . ', id=' . ($footer['id'] ?? 'none'));
        update_option('ai_global_footer', $footer, false);
    }

    return new WP_REST_Response(array(
        'success' => true,
        'navbar_saved' => !empty($navbar),
        'footer_saved' => !empty($footer),
        'navbar_variant' => $navbar['variant'] ?? null,
        'footer_variant' => $footer['variant'] ?? null,
    ), 200);
}

/**
 * Get global navbar and footer blocks
 */
function ai_theme_get_global_blocks() {
    return new WP_REST_Response(array(
        'success' => true,
        'navbar' => get_option('ai_global_navbar', null),
        'footer' => get_option('ai_global_footer', null),
    ), 200);
}

/**
 * Preload widget files for use outside Elementor context (e.g., wp_body_open)
 * This ensures widget classes are available before Elementor's register hook fires
 */
function ai_preload_widget_files() {
    static $loaded = false;
    if ($loaded) return;

    $elementor_dir = get_template_directory() . '/inc/elementor/';
    $widgets_dir = $elementor_dir . 'widgets/';

    // Load base class first
    $base_class_file = $elementor_dir . 'class-ai-widget-base.php';
    if (file_exists($base_class_file) && !class_exists('AITheme\\Elementor\\Widgets\\AI_Widget_Base')) {
        require_once $base_class_file;
    }

    // Load all widget files
    $widget_files = glob($widgets_dir . 'class-*.php');
    foreach ($widget_files as $file) {
        require_once $file;
    }

    $loaded = true;
}

/**
 * Render global navbar block using AI Theme widget
 * Uses variant-specific Elementor widgets for proper rendering
 */
function ai_render_global_navbar() {
    $navbar = get_option('ai_global_navbar', null);
    if (!$navbar) return;

    $content = $navbar['content'] ?? array();
    $variant = $navbar['variant'] ?? 'standard';

    // Debug: log which variant is being rendered
    error_log('[AI Theme] Rendering navbar: variant=' . $variant . ', brandName=' . ($content['brandName'] ?? 'none'));

    // Convert content to Elementor widget settings format
    $settings = ai_convert_navbar_to_widget_settings($content);

    // Normalize variant name: navbar-floating -> floating
    $variant_key = str_replace('navbar-', '', $variant);

    // Map variant to widget class name
    $widget_class_map = array(
        'standard' => 'Navbar_Standard',
        'centered' => 'Navbar_Centered',
        'minimal' => 'Navbar_Minimal',
        'transparent' => 'Navbar_Transparent',
        'floating' => 'Navbar_Floating',
    );

    $class_name = $widget_class_map[$variant_key] ?? 'Navbar_Standard';
    $full_class = 'AITheme\\Elementor\\Widgets\\' . $class_name;

    // Preload widget files so classes are available
    ai_preload_widget_files();

    // Try to use Elementor widget for proper variant rendering
    if (class_exists($full_class)) {
        $widget = new $full_class();
        $widget->render_content_with_settings($settings);
    } else {
        // Fallback to generic navbar render if widget class not available
        ai_render_navbar_widget($variant, $settings);
    }
}

/**
 * Convert navbar content to Elementor widget settings
 */
function ai_convert_navbar_to_widget_settings($content) {
    // Get logo
    $logo_url = '';
    if (!empty($content['logo'])) {
        $logo_url = is_array($content['logo']) ? ($content['logo']['image'] ?? $content['logo']['url'] ?? '') : $content['logo'];
    }

    // Get brand name - prefer logo.text for English content to avoid Russian/localized brandName
    // This handles cases where brandName is in Russian but logo.text is in English
    $language = $content['language'] ?? 'en';
    $logo_text = !empty($content['logo']['text']) ? $content['logo']['text'] : '';
    $brand_name_raw = $content['brandName'] ?? $content['siteName'] ?? '';

    // For English content, prefer logo.text if it looks like English (no Cyrillic)
    if ($language === 'en' && !empty($logo_text) && !preg_match('/[\p{Cyrillic}]/u', $logo_text)) {
        $brand_name = $logo_text;
    } elseif (!empty($brand_name_raw)) {
        $brand_name = $brand_name_raw;
    } elseif (!empty($logo_text)) {
        $brand_name = $logo_text;
    } else {
        $brand_name = get_bloginfo('name');
    }

    // Get navigation items and convert to menu_items format
    $nav_items = $content['navigation'] ?? $content['menuItems'] ?? $content['links'] ?? array();
    $menu_items = array();
    foreach ($nav_items as $item) {
        $menu_items[] = array(
            'title' => $item['label'] ?? $item['text'] ?? $item['title'] ?? '',
            'link' => array(
                'url' => $item['href'] ?? $item['url'] ?? $item['link']['url'] ?? '#',
            ),
        );
    }

    // Get contact info
    $phone = $content['phone'] ?? '';
    $email = $content['email'] ?? '';
    
    // Get CTA button data with localized fallback
    $cta_button = $content['ctaButton'] ?? array();
    $cta_text = $cta_button['text'] ?? '';
    $cta_href = $cta_button['href'] ?? '#contact';
    
    // Fallback for empty CTA text - use localized text
    if (empty($cta_text)) {
        $cta_text = ai_get_text('navbar.ctaButton', ai_get_label('common.contact', 'Contact'));
    }
    
    // Use showCart/showSignIn from content, fallback to 'no' for landings
    $show_cart = !empty($content['showCart']) ? 'yes' : 'no';
    $show_sign_in = !empty($content['showSignIn']) ? 'yes' : 'no';

    return array(
        'logo_text' => $brand_name,
        'logo_image' => $logo_url ? array('url' => $logo_url) : array(),
        'menu_items' => $menu_items,
        'show_cart' => $show_cart,
        'show_sign_in' => $show_sign_in,
        'sign_in_text' => ai_get_text('navbar.login', ai_get_text('account.login', ai_get_label('account.login', 'Log in'))),
        'account_text' => ai_get_text('navbar.myAccount', ai_get_text('profile.myAccount', ai_get_label('account.title', 'Account'))),
        'phone' => $phone,
        'email' => $email,
        // CTA button for landings (shown when cart is hidden)
        'cta_text' => $cta_text,
        'cta_link' => array('url' => $cta_href),
        'show_cta' => !empty($cta_text) ? 'yes' : 'no',
        'cta_url' => array('url' => $cta_href),
    );
}

/**
 * Render navbar widget (standard variant with cart & auth)
 */
function ai_render_navbar_widget($variant, $settings) {
    $unique_id = 'navbar-global-' . uniqid();

    // WooCommerce URLs
    $cart_url = function_exists('wc_get_cart_url') ? wc_get_cart_url() : home_url('/cart/');
    $account_url = function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : home_url('/my-account/');
    $checkout_url = function_exists('wc_get_checkout_url') ? wc_get_checkout_url() : home_url('/checkout/');

    // Cart data
    $cart_count = 0;
    $cart_total = '';
    if (function_exists('WC') && WC()->cart) {
        $cart_count = WC()->cart->get_cart_contents_count();
        $cart_total = WC()->cart->get_cart_total();
    }

    // User state
    $is_logged_in = is_user_logged_in();
    $current_user = wp_get_current_user();

    $show_cart = ($settings['show_cart'] ?? 'yes') === 'yes';
    $show_sign_in = ($settings['show_sign_in'] ?? 'yes') === 'yes';
    ?>
    <nav class="ai-navbar ai-navbar-<?php echo esc_attr($variant); ?> bg-white shadow-md sticky top-0 z-50" id="<?php echo esc_attr($unique_id); ?>">
        <div class="ai-container">
            <div class="flex items-center justify-between h-16">
                <!-- Logo -->
                <a href="<?php echo esc_url(home_url('/')); ?>" class="flex items-center gap-2 flex-shrink-0">
                    <?php if (!empty($settings['logo_image']['url'])) : ?>
                        <img src="<?php echo esc_url($settings['logo_image']['url']); ?>" alt="Logo" class="h-8 object-contain">
                    <?php elseif (!empty($settings['logo_text'])) : ?>
                        <div class="w-8 h-8 bg-gradient-to-br from-blue-600 to-purple-600 rounded-lg flex items-center justify-center text-white font-bold">
                            <?php echo esc_html(mb_substr($settings['logo_text'], 0, 1)); ?>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($settings['logo_text'])) : ?>
                        <span class="text-xl font-bold text-gray-900"><?php echo esc_html($settings['logo_text']); ?></span>
                    <?php endif; ?>
                </a>

                <!-- Desktop Menu -->
                <div class="hidden md:flex items-center gap-6 lg:gap-8">
                    <?php foreach ($settings['menu_items'] ?? array() as $item) : ?>
                        <a href="<?php echo esc_url($item['link']['url'] ?? '#'); ?>" class="text-gray-700 hover:text-primary transition-colors font-medium whitespace-nowrap">
                            <?php echo esc_html($item['title'] ?? ''); ?>
                        </a>
                    <?php endforeach; ?>
                </div>

                <!-- Actions: Cart & Account -->
                <div class="flex items-center gap-2">
                    <?php if ($show_cart) : ?>
                        <!-- Cart with dropdown -->
                        <div class="relative group">
                            <a href="<?php echo esc_url($cart_url); ?>" class="ai-cart-trigger flex items-center gap-2 p-2 hover:bg-gray-100 rounded-lg transition-colors relative">
                                <svg class="w-5 h-5 text-gray-600 group-hover:text-primary transition-colors" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                                <span class="ai-cart-count absolute -top-1 -right-1 bg-primary text-white text-xs w-5 h-5 rounded-full flex items-center justify-center <?php echo $cart_count > 0 ? '' : 'hidden'; ?>"><?php echo esc_html($cart_count); ?></span>
                            </a>

                            <!-- Mini Cart Dropdown -->
                            <div class="ai-mini-cart absolute right-0 top-full mt-2 w-80 bg-white rounded-xl shadow-xl border border-gray-100 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                                <div class="p-4">
                                    <div class="ai-mini-cart-items max-h-64 overflow-y-auto">
                                        <?php if ($cart_count > 0 && function_exists('WC')) : ?>
                                            <?php foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) :
                                                $product = $cart_item['data'];
                                                $quantity = $cart_item['quantity'];
                                                $thumbnail = $product->get_image(array(48, 48), array('class' => 'rounded-lg'));
                                            ?>
                                            <div class="flex items-center gap-3 py-3 border-b border-gray-100 last:border-0">
                                                <div class="w-12 h-12 flex-shrink-0"><?php echo $thumbnail; ?></div>
                                                <div class="flex-1 min-w-0">
                                                    <p class="text-sm font-medium text-gray-900 truncate"><?php echo esc_html($product->get_name()); ?></p>
                                                    <p class="text-sm text-gray-500"><?php echo $quantity; ?> × <?php echo $product->get_price_html(); ?></p>
                                                </div>
                                            </div>
                                            <?php endforeach; ?>
                                        <?php else : ?>
                                            <div class="py-8 text-center text-gray-500">
                                                <svg class="w-12 h-12 mx-auto mb-3 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                                </svg>
                                                <p><?php echo esc_html(ai_get_text('ecommerce.cart.emptyTitle', ai_get_label('cart.empty_title', 'Cart is empty'))); ?></p>
                                            </div>
                                        <?php endif; ?>
                                    </div>

                                    <?php if ($cart_count > 0) : ?>
                                    <div class="mt-4 pt-4 border-t border-gray-100">
                                        <div class="flex items-center justify-between mb-4">
                                            <span class="font-medium text-gray-900"><?php echo esc_html(ai_get_text('ecommerce.cart.total', ai_get_label('cart.total', 'Total:'))); ?></span>
                                            <span class="ai-cart-total font-bold text-lg text-primary"><?php echo $cart_total; ?></span>
                                        </div>
                                        <div class="grid grid-cols-2 gap-2">
                                            <a href="<?php echo esc_url($cart_url); ?>" class="px-4 py-2 text-center text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors">
                                                <?php echo esc_html(ai_get_text('ecommerce.cart.title', ai_get_label('cart.title', 'Cart'))); ?>
                                            </a>
                                            <a href="<?php echo esc_url($checkout_url); ?>" class="px-4 py-2 text-center text-sm font-medium text-white bg-primary hover:opacity-90 rounded-lg transition-opacity">
                                                <?php echo esc_html(ai_get_label('checkout.title', 'Checkout')); ?>
                                            </a>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <?php if ($show_sign_in) : ?>
                        <?php if ($is_logged_in) : ?>
                            <!-- User dropdown (logged in) -->
                            <div class="relative group">
                                <a href="<?php echo esc_url($account_url); ?>" class="flex items-center gap-2 px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-100 rounded-lg transition-colors font-medium">
                                    <div class="w-8 h-8 bg-primary/10 rounded-full flex items-center justify-center text-primary font-semibold">
                                        <?php echo esc_html(mb_strtoupper(mb_substr($current_user->display_name ?: $current_user->user_login, 0, 1))); ?>
                                    </div>
                                    <span class="hidden lg:inline"><?php echo esc_html($settings['account_text'] ?: ai_get_text('profile.myAccount', ai_get_label('account.title', 'Account'))); ?></span>
                                    <svg class="w-4 h-4 hidden lg:block" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                </a>

                                <!-- User Dropdown Menu -->
                                <div class="absolute right-0 top-full mt-2 w-48 bg-white rounded-xl shadow-xl border border-gray-100 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                                    <div class="py-2">
                                        <div class="px-4 py-2 border-b border-gray-100">
                                            <p class="font-medium text-gray-900 truncate"><?php echo esc_html($current_user->display_name ?: $current_user->user_login); ?></p>
                                            <p class="text-sm text-gray-500 truncate"><?php echo esc_html($current_user->user_email); ?></p>
                                        </div>
                                        <a href="<?php echo esc_url($account_url); ?>" class="flex items-center gap-3 px-4 py-2 text-gray-700 hover:bg-gray-50 transition-colors">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"></path></svg>
                                            <?php echo esc_html(ai_get_text('profile.dashboard', ai_get_label('account.dashboard', 'Dashboard'))); ?>
                                        </a>
                                        <a href="<?php echo esc_url(function_exists('wc_get_account_endpoint_url') ? wc_get_account_endpoint_url('orders') : $account_url); ?>" class="flex items-center gap-3 px-4 py-2 text-gray-700 hover:bg-gray-50 transition-colors">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path></svg>
                                            <?php echo esc_html(ai_get_text('profile.orders', ai_get_label('account.orders', 'Orders'))); ?>
                                        </a>
                                        <div class="border-t border-gray-100 mt-2 pt-2">
                                            <a href="<?php echo esc_url(function_exists('wc_logout_url') ? wc_logout_url() : wp_logout_url()); ?>" class="flex items-center gap-3 px-4 py-2 text-red-600 hover:bg-red-50 transition-colors">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg>
                                                <?php echo esc_html(ai_get_text('profile.logOut', ai_get_label('account.logout', 'Log out'))); ?>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php else : ?>
                            <!-- Sign in button -->
                            <a href="<?php echo esc_url($account_url); ?>" class="flex items-center gap-2 px-4 py-2 text-gray-700 hover:text-primary hover:bg-gray-100 rounded-lg transition-colors font-medium">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                                </svg>
                                <span class="hidden lg:inline"><?php echo esc_html($settings['sign_in_text'] ?: ai_get_text('profile.logIn', ai_get_label('account.login', 'Log in'))); ?></span>
                            </a>
                        <?php endif; ?>
                    <?php endif; ?>

                    <?php
                    // CTA Button - shown for landings (when cart is hidden)
                    if (!empty($settings['cta_text']) && !$show_cart) :
                        $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#contact';
                    ?>
                        <a href="<?php echo esc_url($cta_url); ?>" class="hidden md:inline-flex px-4 py-2 bg-primary text-white text-sm rounded-lg hover:opacity-90 transition-opacity font-medium">
                            <?php echo esc_html($settings['cta_text']); ?>
                        </a>
                    <?php endif; ?>

                    <!-- Mobile menu button -->
                    <button class="md:hidden p-2 hover:bg-gray-100 rounded-lg ai-mobile-menu-toggle" aria-label="Toggle menu">
                        <svg class="w-6 h-6 ai-menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                        </svg>
                        <svg class="w-6 h-6 ai-close-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
            </div>

            <!-- Mobile Menu -->
            <div class="ai-mobile-menu md:hidden hidden border-t border-gray-100">
                <div class="py-4 space-y-2">
                    <?php foreach ($settings['menu_items'] ?? array() as $item) : ?>
                        <a href="<?php echo esc_url($item['link']['url'] ?? '#'); ?>" class="block px-4 py-2 text-gray-700 hover:bg-gray-50 hover:text-primary rounded-lg transition-colors font-medium">
                            <?php echo esc_html($item['title'] ?? ''); ?>
                        </a>
                    <?php endforeach; ?>

                    <?php if ($show_cart) : ?>
                    <a href="<?php echo esc_url($cart_url); ?>" class="flex items-center justify-between px-4 py-2 text-gray-700 hover:bg-gray-50 rounded-lg transition-colors font-medium">
                        <span class="flex items-center gap-2">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                            </svg>
                            <?php echo esc_html(ai_get_text('ecommerce.cart.title', ai_get_label('cart.title', 'Cart'))); ?>
                        </span>
                        <?php if ($cart_count > 0) : ?>
                        <span class="bg-primary text-white text-xs px-2 py-1 rounded-full"><?php echo esc_html($cart_count); ?></span>
                        <?php endif; ?>
                    </a>
                    <?php endif; ?>

                    <?php if ($show_sign_in) : ?>
                    <a href="<?php echo esc_url($account_url); ?>" class="flex items-center gap-2 px-4 py-2 text-gray-700 hover:bg-gray-50 rounded-lg transition-colors font-medium">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                        </svg>
                        <?php echo $is_logged_in ? esc_html($settings['account_text'] ?: ai_get_text('profile.myAccount', ai_get_label('account.title', 'Account'))) : esc_html($settings['sign_in_text'] ?: ai_get_text('profile.logIn', ai_get_label('account.login', 'Log in'))); ?>
                    </a>
                    <?php endif; ?>

                    <?php
                    // CTA Button in mobile menu
                    if (!empty($settings['cta_text']) && !$show_cart) :
                        $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#contact';
                    ?>
                    <a href="<?php echo esc_url($cta_url); ?>" class="block mx-4 mt-2 px-4 py-2 bg-primary text-white text-center rounded-lg font-medium">
                        <?php echo esc_html($settings['cta_text']); ?>
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </nav>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        var navbar = document.getElementById('<?php echo esc_js($unique_id); ?>');
        if (navbar) {
            var toggle = navbar.querySelector('.ai-mobile-menu-toggle');
            var menu = navbar.querySelector('.ai-mobile-menu');
            var menuIcon = navbar.querySelector('.ai-menu-icon');
            var closeIcon = navbar.querySelector('.ai-close-icon');

            if (toggle && menu) {
                toggle.addEventListener('click', function() {
                    menu.classList.toggle('hidden');
                    menuIcon.classList.toggle('hidden');
                    closeIcon.classList.toggle('hidden');
                });
            }
        }
    });
    </script>
    <?php
}

/**
 * Render global footer block
 * Uses variant-specific Elementor widgets for proper rendering
 */
function ai_render_global_footer() {
    $footer = get_option('ai_global_footer', null);
    if (!$footer) return;

    $content = $footer['content'] ?? array();
    $variant = $footer['variant'] ?? 'v1';

    // Convert content to Elementor widget settings format
    $settings = ai_convert_footer_to_widget_settings($content);

    // Normalize variant name: footer-v1 -> v1
    $variant_key = str_replace('footer-', '', $variant);
    
    // Map variant to widget class name
    $widget_class_map = array(
        'v1' => 'Footer_V1',
        'centered' => 'Footer_Centered',
        'corporate' => 'Footer_Corporate',
        'mega' => 'Footer_Mega',
        'minimal' => 'Footer_Minimal',
    );
    
    $class_name = $widget_class_map[$variant_key] ?? 'Footer_V1';
    $full_class = 'AITheme\\Elementor\\Widgets\\' . $class_name;

    // Preload widget files so classes are available
    ai_preload_widget_files();

    // Try to use Elementor widget for proper variant rendering
    if (class_exists($full_class)) {
        $widget = new $full_class();
        $widget->render_content_with_settings($settings);
    } else {
        // Fallback to generic footer render
        ai_render_footer_widget_fallback($variant, $settings, $content);
    }
}

/**
 * Convert footer content to Elementor widget settings format
 */
function ai_convert_footer_to_widget_settings($content) {
    // Get logo
    $logo_url = '';
    $logo_text = '';
    if (!empty($content['logo'])) {
        $logo_url = is_array($content['logo']) ? ($content['logo']['image'] ?? $content['logo']['url'] ?? '') : $content['logo'];
        $logo_text = is_array($content['logo']) ? ($content['logo']['text'] ?? '') : '';
    }

    // Get company name - prefer logo.text for English content to avoid Russian/localized brandName
    $language = $content['language'] ?? 'en';
    $brand_name_raw = $content['companyName'] ?? $content['brandName'] ?? '';

    if ($language === 'en' && !empty($logo_text) && !preg_match('/[\p{Cyrillic}]/u', $logo_text)) {
        $company_name = $logo_text;
    } elseif (!empty($brand_name_raw)) {
        $company_name = $brand_name_raw;
    } elseif (!empty($logo_text)) {
        $company_name = $logo_text;
    } else {
        $company_name = get_bloginfo('name');
    }

    $description = $content['description'] ?? $content['tagline'] ?? '';
    $copyright = $content['copyright'] ?? '© ' . date('Y') . ' ' . $company_name;
    
    // Get contact info
    $contacts = $content['contacts'] ?? array();
    $phone = $content['phone'] ?? $contacts['phone'] ?? '';
    $email = $content['email'] ?? $contacts['email'] ?? '';
    $address = $content['address'] ?? $contacts['address'] ?? '';
    
    // Convert columns to Footer Corporate widget format
    // Widget expects: column_title (string), column_links (textarea with "Label|URL\n..." format)
    $columns = $content['columns'] ?? array();
    $widget_columns = array();
    foreach ($columns as $column) {
        $links_lines = array();
        foreach (($column['links'] ?? $column['items'] ?? array()) as $link) {
            $link_url = $link['url'] ?? $link['href'] ?? $link['link']['url'] ?? '#';
            $link_label = $link['text'] ?? $link['label'] ?? $link['title'] ?? '';
            // Convert hash-only links to absolute paths
            if (preg_match('/^#[a-zA-Z]/', $link_url)) {
                $link_url = '/' . $link_url;
            }
            if (!empty($link_label)) {
                $links_lines[] = $link_label . '|' . $link_url;
            }
        }
        $widget_columns[] = array(
            'column_title' => $column['title'] ?? $column['heading'] ?? '',
            'column_links' => implode("\n", $links_lines),
        );
    }
    
    // Convert social links
    $social_links = array();
    foreach (($content['socialLinks'] ?? array()) as $social) {
        // Platform is used by footer-minimal, icon is used by footer-v1
        $platform = strtolower($social['platform'] ?? $social['type'] ?? $social['name'] ?? 'facebook');
        $social_links[] = array(
            'platform' => $platform,
            'icon' => array(
                'value' => ai_map_social_to_fa_icon($platform),
                'library' => 'fa-brands',
            ),
            'link' => array('url' => $social['url'] ?? $social['href'] ?? $social['link'] ?? '#'),
        );
    }

    return array(
        'company_name' => $company_name,
        'company_description' => $description,
        'logo_image' => $logo_url ? array('url' => $logo_url) : array(),
        'logo_text' => $company_name,  // Also set logo_text for widgets that use it
        'copyright_text' => $copyright,
        'copyright' => $copyright,  // Also set copyright for compatibility
        'phone' => $phone,
        'email' => $email,
        'address' => $address,
        'footer_columns' => $widget_columns,  // Widget expects footer_columns, not columns
        'social_links' => $social_links,
        'description' => $description,  // Also set description for widgets that use it
        // CRITICAL: Privacy/Cookies links for footer widgets
        'privacy_link' => array('url' => '/privacy-policy'),
        'cookies_link' => array('url' => '/cookies'),
    );
}

/**
 * Map social platform name to Font Awesome icon class
 */
function ai_map_social_to_fa_icon($platform) {
    $map = array(
        'facebook' => 'fab fa-facebook-f',
        'twitter' => 'fab fa-x-twitter',
        'instagram' => 'fab fa-instagram',
        'linkedin' => 'fab fa-linkedin-in',
        'youtube' => 'fab fa-youtube',
        'telegram' => 'fab fa-telegram',
        'tiktok' => 'fab fa-tiktok',
        'pinterest' => 'fab fa-pinterest',
        'whatsapp' => 'fab fa-whatsapp',
    );
    return $map[$platform] ?? 'fas fa-link';
}

/**
 * Fallback footer rendering when widget class is not available
 */
function ai_render_footer_widget_fallback($variant, $settings, $content) {
    $company_name = $settings['company_name'] ?? get_bloginfo('name');
    $description = $settings['company_description'] ?? '';
    $copyright = $settings['copyright_text'] ?? '© ' . date('Y') . ' ' . $company_name;
    $logo_url = $settings['logo_image']['url'] ?? '';
    $phone = $settings['phone'] ?? '';
    $email = $settings['email'] ?? '';
    $address = $settings['address'] ?? '';
    $columns = $content['columns'] ?? array();
    $social_links = $content['socialLinks'] ?? array();
    ?>
    <footer class="ai-footer ai-footer-<?php echo esc_attr($variant); ?> bg-gray-900 text-white">
        <div class="ai-container py-12">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
                <!-- Company Info -->
                <div class="lg:col-span-1">
                    <div class="flex items-center gap-2 mb-4">
                        <?php if ($logo_url) : ?>
                            <img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr($company_name); ?>" class="h-10 w-auto brightness-0 invert">
                        <?php endif; ?>
                        <span class="text-xl font-bold"><?php echo esc_html($company_name); ?></span>
                    </div>
                    <?php if ($description) : ?>
                        <p class="text-gray-400 mb-4"><?php echo esc_html($description); ?></p>
                    <?php endif; ?>

                    <!-- Contact Info -->
                    <div class="space-y-2 text-sm">
                        <?php if ($phone) : ?>
                            <p class="flex items-center gap-2 text-gray-400">
                                <i class="fas fa-phone text-primary"></i>
                                <a href="tel:<?php echo esc_attr(preg_replace('/[^0-9+]/', '', $phone)); ?>" class="hover:text-white"><?php echo esc_html($phone); ?></a>
                            </p>
                        <?php endif; ?>
                        <?php if ($email) : ?>
                            <p class="flex items-center gap-2 text-gray-400">
                                <i class="fas fa-envelope text-primary"></i>
                                <a href="mailto:<?php echo esc_attr($email); ?>" class="hover:text-white"><?php echo esc_html($email); ?></a>
                            </p>
                        <?php endif; ?>
                        <?php if ($address) : ?>
                            <p class="flex items-start gap-2 text-gray-400">
                                <i class="fas fa-map-marker-alt text-primary mt-1"></i>
                                <span><?php echo esc_html($address); ?></span>
                            </p>
                        <?php endif; ?>
                    </div>

                    <!-- Social Links -->
                    <?php if (!empty($social_links)) : ?>
                        <div class="flex items-center gap-4 mt-4">
                            <?php foreach ($social_links as $social) :
                                $icon = $social['icon'] ?? '';
                                $url = $social['url'] ?? $social['link'] ?? '#';
                                $icon_class = is_array($icon) ? ($icon['value'] ?? 'fas fa-link') : $icon;
                            ?>
                                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener" class="text-gray-400 hover:text-white text-xl">
                                    <i class="<?php echo esc_attr($icon_class); ?>"></i>
                                </a>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Footer Columns -->
                <?php foreach ($columns as $column) :
                    $col_title = $column['title'] ?? $column['heading'] ?? '';
                    $col_links = $column['links'] ?? $column['items'] ?? array();
                ?>
                    <div>
                        <?php if ($col_title) : ?>
                            <h4 class="text-lg font-semibold mb-4"><?php echo esc_html($col_title); ?></h4>
                        <?php endif; ?>
                        <ul class="space-y-2">
                            <?php foreach ($col_links as $link) :
                                $link_text = $link['text'] ?? $link['label'] ?? $link['title'] ?? '';
                                $link_url = $link['url'] ?? $link['href'] ?? $link['link']['url'] ?? '#';

                                // Convert hash-only links to absolute paths
                                if (preg_match('/^#[a-zA-Z]/', $link_url)) {
                                    $link_url = '/' . $link_url;
                                }
                            ?>
                                <li>
                                    <a href="<?php echo esc_url($link_url); ?>" class="text-gray-400 hover:text-white transition-colors">
                                        <?php echo esc_html($link_text); ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Copyright -->
            <div class="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400 text-sm">
                <?php echo esc_html($copyright); ?>
            </div>
        </div>
    </footer>
    <?php
}

/**
 * Render Elementor widget by type with provided settings
 * Used in WooCommerce templates to render saved blocks
 */
function ai_theme_render_widget($widget_type, $settings, $product_context = null) {
    $base_type = explode('-', $widget_type)[0];

    // Use fallback rendering
    return ai_theme_render_widget_fallback($widget_type, $settings, $product_context);
}

/**
 * Fallback widget rendering - uses PHP templates similar to Elementor widgets
 */
function ai_theme_render_widget_fallback($widget_type, $settings, $product_context = null) {
    $base_type = explode('-', $widget_type)[0];

    ob_start();

    switch ($base_type) {
        case 'breadcrumbs':
            ai_render_breadcrumbs_block($settings, $product_context);
            break;

        case 'product':
            // Product block is rendered by WooCommerce template itself
            // Skip it here
            break;

        case 'text':
            ai_render_text_block($settings);
            break;

        case 'features':
            ai_render_features_block($settings);
            break;

        case 'products':
            ai_render_products_block($settings, $product_context);
            break;

        case 'testimonials':
            ai_render_testimonials_block($settings);
            break;

        case 'faq':
            ai_render_faq_block($settings);
            break;

        case 'cta':
            ai_render_cta_block($settings);
            break;

        default:
            echo '<!-- Block type "' . esc_html($widget_type) . '" rendering not implemented -->';
    }

    return ob_get_clean();
}

/**
 * Render breadcrumbs block
 */
function ai_render_breadcrumbs_block($settings, $product_context = null) {
    $heading = $settings['heading'] ?? '';
    $items = $settings['items'] ?? array();

    // If we have product context, build breadcrumbs dynamically
    if ($product_context && is_a($product_context, 'WC_Product')) {
        $items = array(
            array('text' => ai_get_text('common.home', __('Home', 'ai-theme')), 'link' => array('url' => home_url('/'))),
            array('text' => ai_get_text('ecommerce.shop.title', __('Shop', 'ai-theme')), 'link' => array('url' => get_permalink(wc_get_page_id('shop')))),
        );

        $categories = get_the_terms($product_context->get_id(), 'product_cat');
        if ($categories && !is_wp_error($categories)) {
            $main_category = $categories[0];
            $items[] = array(
                'text' => $main_category->name,
                'link' => array('url' => get_term_link($main_category))
            );
        }

        $items[] = array('text' => $product_context->get_name(), 'link' => array('url' => ''));
    }
    ?>
    <nav class="ai-breadcrumbs ai-breadcrumbs-simple w-full bg-gray-50 py-4" aria-label="Breadcrumb">
        <div class="ai-container">
            <?php if ($heading): ?>
            <h2 class="text-lg font-semibold text-gray-900 mb-2"><?php echo esc_html($heading); ?></h2>
            <?php endif; ?>
            <ol class="flex items-center flex-wrap gap-2 text-sm">
                <?php
                $total = count($items);
                foreach ($items as $i => $item):
                ?>
                    <li class="flex items-center gap-2">
                        <?php if ($i < $total - 1 && !empty($item['link']['url'])): ?>
                            <a href="<?php echo esc_url($item['link']['url']); ?>" class="text-gray-600 hover:text-gray-900 transition-colors hover:underline"><?php echo esc_html($item['text']); ?></a>
                            <span class="ai-breadcrumbs-sep text-gray-400" aria-hidden="true">/</span>
                        <?php else: ?>
                            <span class="ai-breadcrumbs-current text-gray-900 font-medium" aria-current="page"><?php echo esc_html($item['text']); ?></span>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ol>
        </div>
    </nav>
    <?php
}

/**
 * Render text block
 */
function ai_render_text_block($settings) {
    $heading = $settings['heading'] ?? '';
    $content = $settings['content'] ?? $settings['description'] ?? '';
    ?>
    <section class="ai-text ai-text-minimal bg-white py-12">
        <div class="ai-container">
            <?php if ($heading): ?>
            <h2 class="text-2xl font-bold text-gray-900 mb-6"><?php echo esc_html($heading); ?></h2>
            <?php endif; ?>
            <?php if ($content): ?>
            <div class="prose prose-lg max-w-none text-gray-600">
                <?php echo wp_kses_post($content); ?>
            </div>
            <?php endif; ?>
        </div>
    </section>
    <?php
}

/**
 * Render features block
 */
function ai_render_features_block($settings) {
    $heading = $settings['heading'] ?? '';
    $description = $settings['description'] ?? '';
    $features = $settings['features'] ?? $settings['items'] ?? array();
    ?>
    <section class="ai-features ai-features-cards bg-gray-50 py-16">
        <div class="ai-container">
            <?php if ($heading): ?>
            <div class="text-center mb-12">
                <h2 class="text-3xl font-bold text-gray-900 mb-4"><?php echo esc_html($heading); ?></h2>
                <?php if ($description): ?>
                <p class="text-lg text-gray-600 max-w-2xl mx-auto"><?php echo esc_html($description); ?></p>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($features as $feature): ?>
                <div class="ai-feature-card bg-white rounded-xl p-6 shadow-sm hover:shadow-md transition-shadow">
                    <?php if (!empty($feature['icon'])): ?>
                    <div class="ai-feature-icon w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mb-4 text-primary">
                        <i class="<?php echo esc_attr(is_array($feature['icon']) ? $feature['icon']['value'] : $feature['icon']); ?> text-xl"></i>
                    </div>
                    <?php endif; ?>
                    <?php if (!empty($feature['title'])): ?>
                    <h3 class="text-lg font-semibold text-gray-900 mb-2"><?php echo esc_html($feature['title']); ?></h3>
                    <?php endif; ?>
                    <?php if (!empty($feature['description'])): ?>
                    <p class="text-gray-600"><?php echo esc_html($feature['description']); ?></p>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
}

/**
 * Render products block (related/recommended products)
 */
function ai_render_products_block($settings, $product_context = null) {
    $heading = $settings['heading'] ?? __('Related Products', 'ai-theme');
    $count = $settings['count'] ?? 4;

    // Get related or featured products
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => $count,
        'post_status' => 'publish',
    );

    // If we have a product context, get related products
    if ($product_context && is_a($product_context, 'WC_Product')) {
        $categories = $product_context->get_category_ids();
        if (!empty($categories)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $categories,
                ),
            );
            $args['post__not_in'] = array($product_context->get_id());
        }
    }

    $products = new WP_Query($args);

    if (!$products->have_posts()) {
        return;
    }
    ?>
    <section class="ai-products ai-products-grid bg-gray-50 py-16">
        <div class="ai-container">
            <?php if ($heading): ?>
            <h2 class="text-2xl font-bold text-gray-900 mb-8"><?php echo esc_html($heading); ?></h2>
            <?php endif; ?>

            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
                <?php while ($products->have_posts()): $products->the_post();
                    $product = wc_get_product(get_the_ID());
                    if (!$product) continue;
                ?>
                <div class="ai-product-card bg-white rounded-xl shadow-sm hover:shadow-lg transition-shadow overflow-hidden">
                    <a href="<?php echo esc_url(get_permalink()); ?>" class="block aspect-square overflow-hidden bg-gray-100">
                        <?php if (has_post_thumbnail()): ?>
                        <img src="<?php echo esc_url(get_the_post_thumbnail_url(get_the_ID(), 'medium')); ?>" alt="<?php echo esc_attr(get_the_title()); ?>" class="w-full h-full object-cover hover:scale-105 transition-transform duration-300">
                        <?php else: ?>
                        <div class="w-full h-full flex items-center justify-center text-gray-400">
                            <i class="fas fa-image text-4xl"></i>
                        </div>
                        <?php endif; ?>
                    </a>
                    <div class="p-4">
                        <a href="<?php echo esc_url(get_permalink()); ?>" class="block">
                            <h3 class="font-semibold text-gray-900 mb-2 line-clamp-2"><?php echo esc_html(get_the_title()); ?></h3>
                        </a>
                        <div class="flex items-baseline gap-2">
                            <?php if ($product->is_on_sale()): ?>
                            <span class="text-lg font-bold text-gray-900"><?php echo wc_price($product->get_sale_price()); ?></span>
                            <span class="text-sm text-gray-500 line-through"><?php echo wc_price($product->get_regular_price()); ?></span>
                            <?php else: ?>
                            <span class="text-lg font-bold text-gray-900"><?php echo wc_price($product->get_price()); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endwhile; wp_reset_postdata(); ?>
            </div>
        </div>
    </section>
    <?php
}

/**
 * Render testimonials block
 */
function ai_render_testimonials_block($settings) {
    $title = $settings['title'] ?? $settings['heading'] ?? '';
    $testimonials = $settings['testimonials'] ?? $settings['items'] ?? array();

    if (empty($testimonials)) return;
    ?>
    <section class="ai-testimonials ai-testimonials-cards bg-white py-16">
        <div class="ai-container">
            <?php if ($title): ?>
            <h2 class="text-3xl font-bold text-gray-900 text-center mb-12"><?php echo esc_html($title); ?></h2>
            <?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($testimonials as $testimonial): ?>
                <div class="ai-testimonial-card bg-gray-50 rounded-xl p-6">
                    <div class="flex items-center mb-4">
                        <?php if (!empty($testimonial['avatar']['url'])): ?>
                        <img src="<?php echo esc_url($testimonial['avatar']['url']); ?>" alt="<?php echo esc_attr($testimonial['name'] ?? ''); ?>" class="w-12 h-12 rounded-full object-cover mr-4">
                        <?php else: ?>
                        <div class="w-12 h-12 rounded-full bg-primary/20 flex items-center justify-center mr-4">
                            <i class="fas fa-user text-primary"></i>
                        </div>
                        <?php endif; ?>
                        <div>
                            <?php if (!empty($testimonial['name'])): ?>
                            <h4 class="font-semibold text-gray-900"><?php echo esc_html($testimonial['name']); ?></h4>
                            <?php endif; ?>
                            <?php if (!empty($testimonial['position'])): ?>
                            <p class="text-sm text-gray-500"><?php echo esc_html($testimonial['position']); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php if (!empty($testimonial['content'])): ?>
                    <p class="text-gray-600 italic">"<?php echo esc_html($testimonial['content']); ?>"</p>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
}

/**
 * Render FAQ block
 */
function ai_render_faq_block($settings) {
    $title = $settings['title'] ?? $settings['heading'] ?? '';
    $faqs = $settings['faqs'] ?? $settings['items'] ?? array();

    if (empty($faqs)) return;
    ?>
    <section class="ai-faq ai-faq-accordion bg-gray-50 py-16">
        <div class="ai-container-narrow">
            <?php if ($title): ?>
            <h2 class="text-3xl font-bold text-gray-900 text-center mb-12"><?php echo esc_html($title); ?></h2>
            <?php endif; ?>

            <div class="space-y-4">
                <?php foreach ($faqs as $i => $faq): ?>
                <div class="ai-faq-item bg-white rounded-lg shadow-sm overflow-hidden">
                    <button onclick="this.parentElement.classList.toggle('open'); this.nextElementSibling.classList.toggle('hidden');"
                            class="w-full px-6 py-4 flex items-center justify-between text-left font-medium text-gray-900 hover:bg-gray-50 transition-colors">
                        <span><?php echo esc_html($faq['question'] ?? ''); ?></span>
                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                    </button>
                    <div class="<?php echo $i === 0 ? '' : 'hidden'; ?> px-6 pb-4 text-gray-600">
                        <?php echo wp_kses_post($faq['answer'] ?? ''); ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
}

/**
 * Render CTA block
 */
function ai_render_cta_block($settings) {
    $heading = $settings['heading'] ?? $settings['title'] ?? '';
    $description = $settings['description'] ?? $settings['subtitle'] ?? '';
    $cta_text = $settings['cta_text'] ?? $settings['button_text'] ?? '';
    $cta_link = $settings['cta_link']['url'] ?? $settings['button_link']['url'] ?? '#';
    ?>
    <section class="ai-cta ai-cta-banner bg-primary py-16">
        <div class="ai-container-narrow text-center">
            <?php if ($heading): ?>
            <h2 class="text-3xl font-bold text-white mb-4"><?php echo esc_html($heading); ?></h2>
            <?php endif; ?>
            <?php if ($description): ?>
            <p class="text-lg text-white/90 mb-8"><?php echo esc_html($description); ?></p>
            <?php endif; ?>
            <?php if ($cta_text): ?>
            <a href="<?php echo esc_url($cta_link); ?>" class="inline-flex items-center bg-white text-primary px-8 py-4 rounded-lg font-semibold hover:bg-gray-100 transition-colors">
                <?php echo esc_html($cta_text); ?>
            </a>
            <?php endif; ?>
        </div>
    </section>
    <?php
}

// ============================================================================
// DUPLICATOR EXPORT REST API
// ============================================================================

/**
 * Register Duplicator REST API routes
 */
function ai_theme_register_duplicator_routes() {
    // Create backup package
    register_rest_route('ai-theme/v1', '/duplicator/create', array(
        'methods'  => 'POST',
        'callback' => 'ai_theme_duplicator_create_package',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ));

    // Get package status
    register_rest_route('ai-theme/v1', '/duplicator/status', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_duplicator_get_status',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ));

    // List available packages
    register_rest_route('ai-theme/v1', '/duplicator/packages', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_duplicator_list_packages',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ));

    // Download package (get URLs)
    register_rest_route('ai-theme/v1', '/duplicator/download/(?P<id>\d+)', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_duplicator_download_package',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ));

    // Stream file download (bypasses .htaccess restrictions)
    register_rest_route('ai-theme/v1', '/duplicator/file/(?P<id>\d+)/(?P<type>archive|installer)', array(
        'methods'  => 'GET',
        'callback' => 'ai_theme_duplicator_stream_file',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ));
}
add_action('rest_api_init', 'ai_theme_register_duplicator_routes');

/**
 * Create a new Duplicator package
 */
function ai_theme_duplicator_create_package($request) {
    // Check if Duplicator is active
    if (!defined('DUPLICATOR_VERSION')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Duplicator plugin is not active',
        ), 400);
    }

    try {
        // Include required Duplicator files
        require_once(DUPLICATOR_PLUGIN_PATH . 'ctrls/ctrl.base.php');
        require_once(DUPLICATOR_PLUGIN_PATH . 'classes/utilities/class.u.php');
        require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');

        // Get package name from request
        $params = $request->get_json_params();
        $package_name = sanitize_text_field($params['package_name'] ?? $params['name'] ?? 'ai-export-' . date('Ymd-His'));

        // Initialize snapshot directory
        DUP_Util::initSnapshotDirectory();

        // Clean up tmp directory before creating new package
        // This fixes "files list file doesn't exist" error
        $tmp_dir = DUP_Settings::getSsdirTmpPath();
        if (is_dir($tmp_dir)) {
            $files = glob($tmp_dir . '/*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    @unlink($file);
                }
            }
        }

        // Create mock POST data as expected by saveActive()
        $mock_post = array(
            'package-name' => $package_name,
            'package-notes' => 'Created via AI Theme REST API',
            'filter-on' => false,
            'filter-paths' => '',
            'filter-exts' => '',
            'dbhost' => '',
            'dbport' => '',
            'dbname' => '',
            'dbuser' => '',
            'dbcharset' => '',
            'dbcollation' => '',
            'secure-on' => false,
            'secure-pass' => '',
            'dbfilter-on' => false,
            'dbtables' => array(),
            'dbcompat' => '',
            'auto-select-components' => 'full', // 'full' or 'database'
        );

        // Create new active package with POST data
        $package = DUP_Package::getActive();
        $package->saveActive($mock_post);

        // Get the updated package with NameHash set
        $package = DUP_Package::getActive();

        // Store for status checking
        update_option('ai_duplicator_active_package', $package->ID, false);
        update_option('ai_duplicator_build_status', 'scanning', false);
        delete_option('ai_duplicator_build_error');

        // Run build synchronously (cron doesn't work reliably in Docker)
        @set_time_limit(600); // 10 minutes
        @ini_set('memory_limit', '512M');

        // Run scanner - creates {NameHash}_files.txt and {NameHash}_dirs.txt
        update_option('ai_duplicator_build_info', 'Scanning files...', false);
        $tmp_dir = DUP_Settings::getSsdirTmpPath();
        $name_hash = $package->NameHash;

        // Log tmp directory contents before scanner
        $files_before = glob($tmp_dir . '/*');
        error_log('Duplicator tmp dir before scan: ' . print_r($files_before, true));

        $report = $package->runScanner();
        $package->saveActiveItem('ScanFile', $package->ScanFile);

        // Log tmp directory contents after scanner
        $files_after = glob($tmp_dir . '/*');
        error_log('Duplicator tmp dir after scan: ' . print_r($files_after, true));
        error_log('Duplicator NameHash: ' . $name_hash);
        error_log('Duplicator ScanFile: ' . $package->ScanFile);

        // Workaround: Duplicator creates _files.txt instead of {NameHash}_files.txt in CLI mode

        // Check for _files.txt (old workaround)
        if (file_exists($tmp_dir . '/_files.txt') && !file_exists($tmp_dir . '/' . $name_hash . '_files.txt')) {
            @rename($tmp_dir . '/_files.txt', $tmp_dir . '/' . $name_hash . '_files.txt');
        }
        if (file_exists($tmp_dir . '/_dirs.txt') && !file_exists($tmp_dir . '/' . $name_hash . '_dirs.txt')) {
            @rename($tmp_dir . '/_dirs.txt', $tmp_dir . '/' . $name_hash . '_dirs.txt');
        }

        // Additional workaround: find any *_files.txt that doesn't match our NameHash and rename it
        $expected_files = $tmp_dir . '/' . $name_hash . '_files.txt';
        if (!file_exists($expected_files)) {
            $all_files = glob($tmp_dir . '/*_files.txt');
            if (!empty($all_files)) {
                @rename($all_files[0], $expected_files);
            }
        }
        $expected_dirs = $tmp_dir . '/' . $name_hash . '_dirs.txt';
        if (!file_exists($expected_dirs)) {
            $all_dirs = glob($tmp_dir . '/*_dirs.txt');
            if (!empty($all_dirs)) {
                @rename($all_dirs[0], $expected_dirs);
            }
        }

        // Build archive
        update_option('ai_duplicator_build_status', 'building', false);
        update_option('ai_duplicator_build_info', 'Building archive...', false);

        $package = DUP_Package::getActive();
        $package->save('zip');
        DUP_Settings::Set('active_package_id', $package->ID);
        DUP_Settings::Save();

        $package->runZipBuild();

        // Check result
        if ($package->Status >= 100) {
            update_option('ai_duplicator_build_status', 'complete', false);
            update_option('ai_duplicator_active_package', $package->ID, false);

            // Fix .htaccess to allow direct download of .zip files
            $htaccess_path = DUP_Settings::getSsdirPath() . '/.htaccess';
            $htaccess_content = "# Duplicator config - modified by AI Theme
Options -Indexes
<Files *.php>
    deny from all
</Files>
<FilesMatch \"\.zip$\">
    Allow from all
    Satisfy any
</FilesMatch>
<IfModule mod_headers.c>
    <FilesMatch \"\.(daf|zip)$\">
        ForceType application/octet-stream
        Header set Content-Disposition attachment
    </FilesMatch>
</IfModule>";
            @file_put_contents($htaccess_path, $htaccess_content);

            return new WP_REST_Response(array(
                'success' => true,
                'packageId' => $package->ID,
                'packageName' => $package_name,
                'nameHash' => $package->NameHash,
                'status' => 'complete',
                'message' => 'Package created successfully',
            ), 200);
        } else {
            update_option('ai_duplicator_build_status', 'error', false);
            update_option('ai_duplicator_build_error', 'Build failed with status: ' . $package->Status, false);

            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'Build failed with status: ' . $package->Status,
            ), 500);
        }

    } catch (Exception $e) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ), 500);
    } catch (Error $e) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ), 500);
    }
}

/**
 * Build package via cron (async)
 */
function ai_duplicator_do_build_package($package_id) {
    if (!defined('DUPLICATOR_VERSION')) return;

    @set_time_limit(0);
    @ini_set('memory_limit', '512M');

    require_once(DUPLICATOR_PLUGIN_PATH . 'ctrls/ctrl.base.php');
    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/utilities/class.u.php');
    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');

    try {
        // Initialize
        DUP_Util::initSnapshotDirectory();

        // Get the active package (with NameHash already set from saveActive)
        $package = DUP_Package::getActive();

        // Verify NameHash is set
        if (empty($package->NameHash)) {
            throw new Exception('Package NameHash is not set. Package may not have been initialized properly.');
        }

        // First run scanner
        update_option('ai_duplicator_build_status', 'scanning', false);
        update_option('ai_duplicator_build_info', 'Scanning files (NameHash: ' . $package->NameHash . ')', false);

        // Run scanner - this creates {NameHash}_files.txt and {NameHash}_dirs.txt
        $report = $package->runScanner();
        $package->saveActiveItem('ScanFile', $package->ScanFile);

        // Workaround: Duplicator bug creates _files.txt instead of {NameHash}_files.txt
        // when running via CLI/cron. Rename the files to correct names.
        $tmp_dir = DUP_Settings::getSsdirTmpPath();
        $name_hash = $package->NameHash;
        if (file_exists($tmp_dir . '/_files.txt') && !file_exists($tmp_dir . '/' . $name_hash . '_files.txt')) {
            @rename($tmp_dir . '/_files.txt', $tmp_dir . '/' . $name_hash . '_files.txt');
        }
        if (file_exists($tmp_dir . '/_dirs.txt') && !file_exists($tmp_dir . '/' . $name_hash . '_dirs.txt')) {
            @rename($tmp_dir . '/_dirs.txt', $tmp_dir . '/' . $name_hash . '_dirs.txt');
        }

        // Refresh package after scanner
        $package = DUP_Package::getActive();

        // Now build
        update_option('ai_duplicator_build_status', 'building', false);
        update_option('ai_duplicator_build_info', 'Building archive...', false);

        // Save package to DB and start logging
        $package->save('zip');
        DUP_Settings::Set('active_package_id', $package->ID);
        DUP_Settings::Save();

        // Build the archive (includes database dump, file archive, and installer)
        $package->runZipBuild();

        // Check if successful (Status >= 100 means complete)
        if ($package->Status >= 100) {
            update_option('ai_duplicator_build_status', 'complete', false);
            update_option('ai_duplicator_active_package', $package->ID, false);
            update_option('ai_duplicator_build_info', 'Package created successfully', false);
        } else {
            update_option('ai_duplicator_build_status', 'error', false);
            update_option('ai_duplicator_build_error', 'Build failed with status: ' . $package->Status, false);
        }

    } catch (Exception $e) {
        update_option('ai_duplicator_build_status', 'error', false);
        update_option('ai_duplicator_build_error', $e->getMessage(), false);
    } catch (Error $e) {
        update_option('ai_duplicator_build_status', 'error', false);
        update_option('ai_duplicator_build_error', $e->getMessage(), false);
    }
}
add_action('ai_duplicator_build_package', 'ai_duplicator_do_build_package');

/**
 * Get current build status
 */
function ai_theme_duplicator_get_status($request) {
    $status = get_option('ai_duplicator_build_status', 'idle');
    $package_id = get_option('ai_duplicator_active_package', null);
    $error = get_option('ai_duplicator_build_error', null);

    $response = array(
        'success' => true,
        'status' => $status,
        'packageId' => $package_id,
    );

    if ($error && $status === 'error') {
        $response['error'] = $error;
    }

    // Add info from option if available
    $info = get_option('ai_duplicator_build_info', null);
    if ($info) {
        $response['info'] = $info;
    }

    // If complete, get download URLs
    if ($status === 'complete' && $package_id) {
        if (defined('DUPLICATOR_VERSION')) {
            require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');
            $package = DUP_Package::getByID($package_id);
            if ($package) {
                // Build URLs using Duplicator settings (works for both Lite and Pro)
                require_once(DUPLICATOR_PLUGIN_PATH . 'classes/class.settings.php');
                $backup_url = DUP_Settings::getSsdirUrl() . '/';
                $name_hash = $package->NameHash;
                $archive_name = $name_hash . '_archive.zip';
                $installer_name = $name_hash . '_installer.php';
                $response['package'] = array(
                    'id' => $package->ID,
                    'name' => $package->Name,
                    'created' => $package->Created,
                    'nameHash' => $name_hash,
                    'archiveFile' => $archive_name,
                    'installerFile' => $installer_name,
                    'archiveUrl' => $backup_url . $archive_name,
                    'installerUrl' => $backup_url . $installer_name,
                );
            }
        }
    }

    return new WP_REST_Response($response, 200);
}

/**
 * List all available packages
 */
function ai_theme_duplicator_list_packages($request) {
    if (!defined('DUPLICATOR_VERSION')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Duplicator plugin is not active',
        ), 400);
    }

    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');
    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/class.settings.php');

    global $wpdb;
    $table = $wpdb->prefix . 'duplicator_packages';

    $packages = $wpdb->get_results(
        "SELECT id, name, created, status FROM {$table} WHERE status >= 100 ORDER BY created DESC LIMIT 10"
    );

    $result = array();
    // Use Duplicator settings to get correct backup URL (works for both Lite and Pro)
    $snapshots_url = DUP_Settings::getSsdirUrl() . '/';
    foreach ($packages as $pkg) {
        $package = DUP_Package::getByID($pkg->id);
        if ($package) {
            // Build URLs manually using package NameHash
            $name_hash = $package->NameHash;
            $archive_name = $name_hash . '_archive.zip';
            $installer_name = $name_hash . '_installer.php';

            $result[] = array(
                'id' => $package->ID,
                'name' => $package->Name,
                'created' => $package->Created,
                'size' => $package->Archive->Size ?? 0,
                'archiveUrl' => $snapshots_url . $archive_name,
                'installerUrl' => $snapshots_url . $installer_name,
            );
        }
    }

    return new WP_REST_Response(array(
        'success' => true,
        'packages' => $result,
    ), 200);
}

/**
 * Download package archive
 */
function ai_theme_duplicator_download_package($request) {
    $package_id = intval($request->get_param('id'));

    if (!defined('DUPLICATOR_VERSION')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Duplicator plugin is not active',
        ), 400);
    }

    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');

    $package = DUP_Package::getByID($package_id);
    if (!$package) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Package not found',
        ), 404);
    }

    // Return download URLs
    return new WP_REST_Response(array(
        'success' => true,
        'package' => array(
            'id' => $package->ID,
            'name' => $package->Name,
            'archiveFile' => $package->Archive->File,
            'installerFile' => $package->Installer->File,
            'archiveUrl' => $package->getLocalPackageURL(DUP_PackageFileType::Archive),
            'installerUrl' => $package->getLocalPackageURL(DUP_PackageFileType::Installer),
        ),
    ), 200);
}

/**
 * Stream file download (bypasses .htaccess restrictions)
 */
function ai_theme_duplicator_stream_file($request) {
    $package_id = intval($request->get_param('id'));
    $file_type = $request->get_param('type'); // 'archive' or 'installer'

    if (!defined('DUPLICATOR_VERSION')) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Duplicator plugin is not active',
        ), 400);
    }

    require_once(DUPLICATOR_PLUGIN_PATH . 'classes/package/class.pack.php');

    $package = DUP_Package::getByID($package_id);
    if (!$package) {
        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'Package not found',
        ), 404);
    }

    // Get file path
    $backup_dir = DUP_Settings::getSsdirPath();
    $name_hash = $package->NameHash;

    if ($file_type === 'archive') {
        $filename = $name_hash . '_archive.zip';
        $content_type = 'application/zip';
    } else {
        // Duplicator renames installer.php to installer.php.bak for security
        // Check both variants
        $filename = $name_hash . '_installer.php';
        $filename_bak = $name_hash . '_installer.php.bak';
        $content_type = 'application/octet-stream';
    }

    $file_path = $backup_dir . '/' . $filename;

    // For installer, check .bak variant first (Duplicator security rename)
    if ($file_type === 'installer' && !file_exists($file_path) && file_exists($backup_dir . '/' . $filename_bak)) {
        $file_path = $backup_dir . '/' . $filename_bak;
        // Keep original .php filename for download (remove .bak)
        // User needs installer.php, not installer.php.bak
    }

    // Debug logging
    error_log("Duplicator stream_file: package_id={$package_id}, type={$file_type}");
    error_log("Duplicator stream_file: backup_dir={$backup_dir}");
    error_log("Duplicator stream_file: name_hash={$name_hash}");
    error_log("Duplicator stream_file: file_path={$file_path}");
    error_log("Duplicator stream_file: file_exists=" . (file_exists($file_path) ? 'yes' : 'no'));

    if (!file_exists($file_path)) {
        // List files in backup directory for debugging
        $files_in_dir = scandir($backup_dir);
        error_log("Duplicator stream_file: files in backup_dir: " . implode(', ', array_slice($files_in_dir, 0, 20)));

        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'File not found: ' . $filename,
            'debug' => array(
                'backup_dir' => $backup_dir,
                'name_hash' => $name_hash,
                'file_path' => $file_path,
            ),
        ), 404);
    }

    // Clear any output buffers
    while (ob_get_level()) {
        ob_end_clean();
    }

    // For installer.php - patch to remove password protection
    if ($file_type === 'installer') {
        $content = file_get_contents($file_path);

        // Patch Duplicator password protection - various patterns for different versions
        // Pattern 1: 'secure_on' => true or 'secure_on' => 1
        $content = preg_replace(
            "/(['\"]secure_on['\"])\s*=>\s*(true|1|'1'|\"1\")/i",
            "'secure_on' => false",
            $content
        );

        // Pattern 2: 'secure_pass' => 'hash...'
        $content = preg_replace(
            "/(['\"]secure_pass['\"])\s*=>\s*['\"][^'\"]*['\"]/i",
            "'secure_pass' => ''",
            $content
        );

        // Pattern 3: $GLOBALS['DUPX_AC']->secure_on = true
        $content = preg_replace(
            "/(\\\$GLOBALS\s*\[\s*['\"]DUPX_AC['\"]\s*\]\s*->\s*secure_on\s*=\s*)(true|1)/i",
            "$1false",
            $content
        );

        // Pattern 4: $GLOBALS['DUPX_AC']->secure_pass = 'hash'
        $content = preg_replace(
            "/(\\\$GLOBALS\s*\[\s*['\"]DUPX_AC['\"]\s*\]\s*->\s*secure_pass\s*=\s*)['\"][^'\"]*['\"]/i",
            "$1''",
            $content
        );

        // Pattern 5: For newer Duplicator versions - DUPX_Security class
        $content = preg_replace(
            "/(['\"]archiveSecure['\"])\s*=>\s*(true|1|'1'|\"1\")/i",
            "'archiveSecure' => false",
            $content
        );

        $content = preg_replace(
            "/(['\"]archivePassword['\"])\s*=>\s*['\"][^'\"]*['\"]/i",
            "'archivePassword' => ''",
            $content
        );

        // Pattern 6: Patch archive filename to 'archive.zip' (for bundle compatibility)
        // Duplicator uses original filename like 'export_abc123_20241220_archive.zip'
        // but our bundle renames it to 'archive.zip'
        $content = preg_replace(
            "/(['\"])([^'\"]*_archive\.zip)\1/",
            "'archive.zip'",
            $content
        );

        // Set headers with patched content length
        $file_size = strlen($content);
        header('Content-Type: ' . $content_type);
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . $file_size);
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');

        echo $content;
        exit;
    }

    // For archive.zip - stream as-is
    $file_size = filesize($file_path);
    header('Content-Type: ' . $content_type);
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . $file_size);
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: no-cache');

    readfile($file_path);
    exit;
}

// ============================================================================
// COOKIE CONSENT MODAL
// ============================================================================

/**
 * Render cookie consent modal in footer
 * Shows a modal asking user to accept cookies on first visit
 */
function ai_theme_render_cookie_consent() {
    // Use AI-generated siteTexts for cookie consent (match generator keys)
    $t = [
        'title' => ai_get_text('cookies.bannerTitle', 'We use cookies'),
        'description' => ai_get_text('cookies.bannerDescription', 'We use cookies to improve website performance, analyze traffic, and personalize content. By continuing to use the site, you agree to our cookie policy.'),
        'accept' => ai_get_text('cookies.acceptButton', 'Accept'),
        'learnMore' => ai_get_text('cookies.learnMoreLink', 'Learn more about cookies'),
        'agreement' => ai_get_text('cookies.consentNote', 'By clicking "Accept", you agree to the use of cookies'),
    ];

    // Get cookies page URL (try to find page with slug 'cookies' or 'cookie-policy')
    $cookies_page = get_page_by_path('cookies');
    if (!$cookies_page) {
        $cookies_page = get_page_by_path('cookie-policy');
    }
    $cookies_url = $cookies_page ? get_permalink($cookies_page) : '#';
    ?>

    <!-- Cookie Consent Modal -->
    <div id="ai-cookie-consent" class="fixed inset-0 z-[9999] hidden items-center justify-center p-4" style="background: rgba(0,0,0,0.5); backdrop-filter: blur(4px);">
        <div class="ai-cookie-modal-content bg-white rounded-2xl shadow-2xl max-w-md w-full p-6 sm:p-8 transform transition-all duration-300">
            <!-- Cookie Icon -->
            <div class="flex justify-center mb-4">
                <div class="w-16 h-16 bg-amber-100 rounded-full flex items-center justify-center">
                    <svg class="w-10 h-10 text-amber-600" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M10 2a8 8 0 100 16 8 8 0 000-16zM7 9a1 1 0 100-2 1 1 0 000 2zm6 0a1 1 0 100-2 1 1 0 000 2zm-3 4a1 1 0 100-2 1 1 0 000 2zm3-1a1 1 0 11-2 0 1 1 0 012 0z"/>
                    </svg>
                </div>
            </div>

            <!-- Title -->
            <h2 class="text-2xl font-bold text-gray-900 text-center mb-3">
                <?php echo esc_html($t['title']); ?>
            </h2>

            <!-- Description -->
            <p class="text-gray-600 text-center mb-6 leading-relaxed">
                <?php echo esc_html($t['description']); ?>
            </p>

            <!-- Buttons -->
            <div class="flex flex-col gap-3">
                <button id="ai-cookie-accept" class="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors duration-200 shadow-lg hover:shadow-xl">
                    <?php echo esc_html($t['accept']); ?>
                </button>

                <a href="<?php echo esc_url($cookies_url); ?>" class="w-full text-center text-blue-600 hover:text-blue-700 font-medium py-2 px-6 rounded-lg transition-colors duration-200 hover:bg-blue-50">
                    <?php echo esc_html($t['learnMore']); ?>
                </a>
            </div>

            <!-- Agreement text -->
            <p class="text-xs text-gray-500 text-center mt-4">
                <?php echo esc_html($t['agreement']); ?>
            </p>
        </div>
    </div>

    <style>
        @keyframes aiCookieSlideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        #ai-cookie-consent.show {
            display: flex !important;
        }

        #ai-cookie-consent.show .ai-cookie-modal-content {
            animation: aiCookieSlideUp 0.3s ease-out;
        }
    </style>
    <?php
}
add_action('wp_footer', 'ai_theme_render_cookie_consent', 99);
