/**
 * AI Theme JavaScript
 * Handles interactivity for theme blocks
 */

(function() {
    'use strict';

    // Mobile menu toggle
    function initMobileMenu() {
        const toggleBtns = document.querySelectorAll('[data-mobile-toggle]');
        const mobileMenus = document.querySelectorAll('[data-mobile-menu]');

        toggleBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const targetId = this.getAttribute('data-mobile-toggle');
                const menu = document.querySelector(`[data-mobile-menu="${targetId}"]`);

                if (menu) {
                    menu.classList.toggle('hidden');
                    this.setAttribute('aria-expanded',
                        menu.classList.contains('hidden') ? 'false' : 'true'
                    );
                }
            });
        });
    }

    // Smooth scroll for anchor links
    // Supports: #id, /#id, #block-type (via data-block-type or .ai-{type} class)
    function initSmoothScroll() {
        // Handle both #hash and /#hash links
        document.querySelectorAll('a[href^="#"], a[href^="/#"]').forEach(anchor => {
            anchor.addEventListener('click', function(e) {
                const href = this.getAttribute('href');
                if (href === '#' || href === '/#') return;

                // Extract hash from href (handles both #hash and /#hash)
                let hash;
                if (href.startsWith('/#')) {
                    // Link like /#features - check if we're on homepage
                    const isHomepage = window.location.pathname === '/' || window.location.pathname === '';
                    if (!isHomepage) {
                        // Not on homepage - let browser navigate to /#hash
                        return;
                    }
                    hash = href.substring(2); // Remove /#
                } else {
                    hash = href.substring(1); // Remove #
                }

                const target = findScrollTarget(hash);

                if (target) {
                    e.preventDefault();

                    // Account for fixed navbar height
                    const navbar = document.querySelector('.ai-navbar, [data-navbar-sticky]');
                    const navbarHeight = navbar ? navbar.offsetHeight : 0;
                    const targetPosition = target.getBoundingClientRect().top + window.pageYOffset - navbarHeight - 20;

                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });

                    // Update URL without triggering scroll
                    history.pushState(null, '', '#' + hash);
                }
            });
        });

        // Handle initial page load with hash (works after navigation from other pages)
        if (window.location.hash) {
            setTimeout(() => {
                const hash = window.location.hash.substring(1);
                const target = findScrollTarget(hash);
                if (target) {
                    const navbar = document.querySelector('.ai-navbar, [data-navbar-sticky]');
                    const navbarHeight = navbar ? navbar.offsetHeight : 0;
                    const targetPosition = target.getBoundingClientRect().top + window.pageYOffset - navbarHeight - 20;
                    window.scrollTo({ top: targetPosition, behavior: 'smooth' });
                }
            }, 100);
        }
    }

    // Find scroll target by hash - tries multiple selectors with smart fallbacks
    function findScrollTarget(hash) {
        // Define fallback chains for common action sections
        // newsletter -> contact -> cta (for subscription/contact actions)
        // contact -> newsletter -> cta (for contact actions)
        const fallbackChains = {
            'newsletter': ['newsletter', 'contact', 'cta'],
            'contact': ['contact', 'newsletter', 'cta'],
            'cta': ['cta', 'newsletter', 'contact'],
            'subscribe': ['newsletter', 'contact', 'cta'],
        };

        // Get the chain of hashes to try (original hash + fallbacks)
        const hashesToTry = fallbackChains[hash] || [hash];

        for (const currentHash of hashesToTry) {
            // 1. Try exact id match
            let target = document.getElementById(currentHash);
            if (target) return target;

            // 2. Try data-block-type attribute (e.g., #contact -> [data-block-type="contact"])
            target = document.querySelector(`[data-block-type="${currentHash}"]`);
            if (target) return target;

            // 3. Try ai-{hash} class (e.g., #contact -> .ai-contact)
            target = document.querySelector(`.ai-${currentHash}`);
            if (target) return target;

            // 4. Try data-section-id attribute
            target = document.querySelector(`[data-section-id="${currentHash}"]`);
            if (target) return target;
        }

        return null;
    }

    // Navbar scroll behavior
    function initNavbarScroll() {
        const navbar = document.querySelector('[data-navbar-sticky]');
        if (!navbar) return;

        let lastScroll = 0;
        const scrollThreshold = 100;

        window.addEventListener('scroll', function() {
            const currentScroll = window.pageYOffset;

            // Add shadow when scrolled
            if (currentScroll > 10) {
                navbar.classList.add('shadow-md');
            } else {
                navbar.classList.remove('shadow-md');
            }

            // Hide/show on scroll (optional)
            if (navbar.hasAttribute('data-navbar-hide-on-scroll')) {
                if (currentScroll > lastScroll && currentScroll > scrollThreshold) {
                    navbar.style.transform = 'translateY(-100%)';
                } else {
                    navbar.style.transform = 'translateY(0)';
                }
            }

            lastScroll = currentScroll;
        });
    }

    // FAQ accordion
    function initFaqAccordion() {
        const faqItems = document.querySelectorAll('[data-faq-item]');

        faqItems.forEach(item => {
            const question = item.querySelector('[data-faq-question]');
            const icon = item.querySelector('[data-faq-icon]');

            if (question) {
                question.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    const isOpen = item.classList.contains('active');

                    // Close all other items (accordion behavior)
                    faqItems.forEach(otherItem => {
                        if (otherItem !== item) {
                            otherItem.classList.remove('active');
                            const otherIcon = otherItem.querySelector('[data-faq-icon]');
                            if (otherIcon) otherIcon.style.transform = 'rotate(0deg)';
                        }
                    });

                    // Toggle current item
                    item.classList.toggle('active');
                    if (icon) {
                        icon.style.transform = isOpen ? 'rotate(0deg)' : 'rotate(180deg)';
                    }
                });
            }
        });
    }

    // Modal functionality
    function initModals() {
        // Open modal
        document.querySelectorAll('[data-modal-open]').forEach(trigger => {
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                const modalId = this.getAttribute('data-modal-open');
                const modal = document.querySelector(`[data-modal="${modalId}"]`);
                if (modal) {
                    modal.classList.remove('hidden');
                    document.body.style.overflow = 'hidden';
                }
            });
        });

        // Close modal
        document.querySelectorAll('[data-modal-close]').forEach(closeBtn => {
            closeBtn.addEventListener('click', function() {
                const modal = this.closest('[data-modal]');
                if (modal) {
                    modal.classList.add('hidden');
                    document.body.style.overflow = '';
                }
            });
        });

        // Close on overlay click
        document.querySelectorAll('[data-modal]').forEach(modal => {
            modal.addEventListener('click', function(e) {
                if (e.target === this) {
                    this.classList.add('hidden');
                    document.body.style.overflow = '';
                }
            });
        });

        // Close on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                document.querySelectorAll('[data-modal]:not(.hidden)').forEach(modal => {
                    modal.classList.add('hidden');
                    document.body.style.overflow = '';
                });
            }
        });
    }

    // Testimonials slider (basic)
    function initTestimonialsSlider() {
        const sliders = document.querySelectorAll('[data-testimonials-slider]');

        sliders.forEach(slider => {
            const slides = slider.querySelectorAll('[data-slide]');
            const prevBtn = slider.querySelector('[data-slide-prev]');
            const nextBtn = slider.querySelector('[data-slide-next]');
            const dots = slider.querySelectorAll('[data-slide-dot]');
            let currentIndex = 0;

            function showSlide(index) {
                slides.forEach((slide, i) => {
                    slide.classList.toggle('hidden', i !== index);
                });
                dots.forEach((dot, i) => {
                    dot.classList.toggle('bg-primary', i === index);
                    dot.classList.toggle('bg-gray-300', i !== index);
                });
            }

            if (prevBtn) {
                prevBtn.addEventListener('click', () => {
                    currentIndex = (currentIndex - 1 + slides.length) % slides.length;
                    showSlide(currentIndex);
                });
            }

            if (nextBtn) {
                nextBtn.addEventListener('click', () => {
                    currentIndex = (currentIndex + 1) % slides.length;
                    showSlide(currentIndex);
                });
            }

            dots.forEach((dot, i) => {
                dot.addEventListener('click', () => {
                    currentIndex = i;
                    showSlide(currentIndex);
                });
            });

            // Auto-advance (optional)
            if (slider.hasAttribute('data-autoplay')) {
                setInterval(() => {
                    currentIndex = (currentIndex + 1) % slides.length;
                    showSlide(currentIndex);
                }, 5000);
            }
        });
    }

    // Animate elements on scroll
    function initScrollAnimations() {
        const animatedElements = document.querySelectorAll('[data-animate-on-scroll]');

        if (!animatedElements.length) return;

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const animation = entry.target.getAttribute('data-animate-on-scroll');
                    entry.target.classList.add(`animate-${animation}`);
                    observer.unobserve(entry.target);
                }
            });
        }, {
            threshold: 0.1
        });

        animatedElements.forEach(el => observer.observe(el));
    }

    // Cookie consent modal
    function initCookieConsent() {
        const COOKIE_CONSENT_KEY = 'ai_cookie_consent_accepted';
        const modal = document.getElementById('ai-cookie-consent');
        const acceptBtn = document.getElementById('ai-cookie-accept');

        if (!modal) return;

        // Check if consent was already given
        const consentGiven = localStorage.getItem(COOKIE_CONSENT_KEY);

        if (!consentGiven) {
            // Show modal with delay for better UX
            setTimeout(() => {
                modal.classList.add('show');
                document.body.style.overflow = 'hidden';
            }, 1000);
        }

        // Accept button handler
        if (acceptBtn) {
            acceptBtn.addEventListener('click', function() {
                localStorage.setItem(COOKIE_CONSENT_KEY, 'true');
                modal.classList.remove('show');
                document.body.style.overflow = '';
            });
        }

        // Close on overlay click (outside modal content)
        modal.addEventListener('click', function(e) {
            if (e.target === modal) {
                localStorage.setItem(COOKIE_CONSENT_KEY, 'true');
                modal.classList.remove('show');
                document.body.style.overflow = '';
            }
        });
    }

    // Counter animation for stats
    function initCounters() {
        const counters = document.querySelectorAll('[data-counter]');

        if (!counters.length) return;

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const counter = entry.target;
                    const target = parseInt(counter.getAttribute('data-counter'));
                    const duration = parseInt(counter.getAttribute('data-counter-duration') || 2000);
                    const start = 0;
                    const startTime = performance.now();

                    function updateCounter(currentTime) {
                        const elapsed = currentTime - startTime;
                        const progress = Math.min(elapsed / duration, 1);
                        const current = Math.floor(progress * (target - start) + start);
                        counter.textContent = current.toLocaleString();

                        if (progress < 1) {
                            requestAnimationFrame(updateCounter);
                        } else {
                            counter.textContent = target.toLocaleString();
                        }
                    }

                    requestAnimationFrame(updateCounter);
                    observer.unobserve(counter);
                }
            });
        }, {
            threshold: 0.5
        });

        counters.forEach(counter => observer.observe(counter));
    }

    // Contact form handling (demo mode - simulates submission)
    function initContactForms() {
        const contactSections = document.querySelectorAll('.ai-contact');

        contactSections.forEach(section => {
            const button = section.querySelector('.ai-contact-button');
            if (!button) return;

            button.addEventListener('click', function(e) {
                e.preventDefault();

                // Find form inputs within this section
                const nameInput = section.querySelector('input[type="text"]');
                const emailInput = section.querySelector('input[type="email"]');
                const messageInput = section.querySelector('textarea');
                const privacyCheckbox = section.querySelector('input[type="checkbox"]');

                // Validation
                const errors = [];

                if (!nameInput || !nameInput.value.trim()) {
                    errors.push('name');
                    if (nameInput) nameInput.classList.add('border-red-500');
                }

                if (!emailInput || !emailInput.value.trim() || !isValidEmail(emailInput.value)) {
                    errors.push('email');
                    if (emailInput) emailInput.classList.add('border-red-500');
                }

                if (!messageInput || !messageInput.value.trim()) {
                    errors.push('message');
                    if (messageInput) messageInput.classList.add('border-red-500');
                }

                if (privacyCheckbox && !privacyCheckbox.checked) {
                    errors.push('privacy');
                    privacyCheckbox.parentElement?.classList.add('border-red-500');
                }

                // Remove error styles on input
                [nameInput, emailInput, messageInput].forEach(input => {
                    if (input) {
                        input.addEventListener('input', function() {
                            this.classList.remove('border-red-500');
                        }, { once: true });
                    }
                });

                if (privacyCheckbox) {
                    privacyCheckbox.addEventListener('change', function() {
                        this.parentElement?.classList.remove('border-red-500');
                    }, { once: true });
                }

                if (errors.length > 0) {
                    // Shake button to indicate error
                    button.classList.add('animate-shake');
                    setTimeout(() => button.classList.remove('animate-shake'), 500);
                    return;
                }

                // Show loading state
                const originalText = button.innerHTML;
                button.disabled = true;
                button.innerHTML = '<span class="animate-spin inline-block w-5 h-5 border-2 border-white border-t-transparent rounded-full"></span>';

                // Simulate sending (1.5s delay)
                setTimeout(() => {
                    // Show success message
                    showFormSuccess(section, button, originalText);

                    // Clear form
                    if (nameInput) nameInput.value = '';
                    if (emailInput) emailInput.value = '';
                    if (messageInput) messageInput.value = '';
                    if (privacyCheckbox) privacyCheckbox.checked = false;
                }, 1500);
            });
        });
    }

    // Email validation helper
    function isValidEmail(email) {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
    }

    // Show success message after form submission
    function showFormSuccess(section, button, originalText) {
        // Get localized success message or use default
        const successTitle = window.aiThemeTexts?.formSuccessTitle || 'Thank you!';
        const successMessage = window.aiThemeTexts?.formSuccessMessage || 'Your message has been sent. We will contact you soon.';

        // Create success overlay
        const formContainer = button.closest('.space-y-4') || button.parentElement;
        const originalDisplay = formContainer.innerHTML;

        formContainer.innerHTML = `
            <div class="text-center py-8 animate-fade-in">
                <div class="w-16 h-16 mx-auto mb-4 rounded-full bg-green-100 flex items-center justify-center">
                    <svg class="w-8 h-8 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                </div>
                <h3 class="text-xl font-semibold text-gray-900 mb-2">${successTitle}</h3>
                <p class="text-gray-600">${successMessage}</p>
                <button class="mt-6 px-6 py-2 text-sm text-blue-600 hover:text-blue-700 font-medium" data-reset-form>
                    ← Send another message
                </button>
            </div>
        `;

        // Handle reset button
        const resetBtn = formContainer.querySelector('[data-reset-form]');
        if (resetBtn) {
            resetBtn.addEventListener('click', function() {
                formContainer.innerHTML = originalDisplay;
                // Re-initialize the form
                initContactForms();
            });
        }
    }

    // Initialize all components
    function init() {
        initMobileMenu();
        initSmoothScroll();
        initNavbarScroll();
        initFaqAccordion();
        initModals();
        initTestimonialsSlider();
        initScrollAnimations();
        initCounters();
        initCookieConsent();
        initContactForms();
    }

    // Run on DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
