<?php
/**
 * Template Name: Blog Page
 * Template for blog page with slug 'blog'
 *
 * This template displays blog posts listing.
 */

get_header();

// Get all categories for filter
$categories = get_categories([
    'hide_empty' => true,
    'orderby' => 'name',
]);

// Pagination
$paged = get_query_var('paged') ? get_query_var('paged') : 1;

// Query posts
$args = [
    'post_type' => 'post',
    'post_status' => 'publish',
    'posts_per_page' => 9,
    'paged' => $paged,
];

$blog_query = new WP_Query($args);
?>

<div class="min-h-screen bg-gray-50" data-theme="default">
    <!-- Navbar is rendered by header.php -->

    <!-- Page Header -->
    <div class="bg-gradient-to-r from-gray-900 to-gray-800 text-white py-16">
        <div class="ai-container text-center">
            <h1 class="text-4xl md:text-5xl font-bold mb-4"><?php echo esc_html(ai_get_text('blogPage.title', 'Blog')); ?></h1>
            <p class="text-xl text-gray-300 max-w-2xl mx-auto">
                <?php echo esc_html(ai_get_text('blogPage.subtitle', 'Latest news, articles, and updates from our team')); ?>
            </p>
        </div>
    </div>

    <!-- Breadcrumbs -->
    <div class="bg-white border-b border-gray-200">
        <div class="ai-container py-4">
            <nav class="flex items-center gap-2 text-sm text-gray-500">
                <a href="<?php echo home_url('/'); ?>" class="hover:text-primary transition-colors"><?php echo esc_html(ai_get_text('common.home', 'Home')); ?></a>
                <span>/</span>
                <span class="text-gray-900 font-medium"><?php echo esc_html(ai_get_text('blogPage.title', 'Blog')); ?></span>
            </nav>
        </div>
    </div>

    <div class="ai-container py-12">

        <!-- Category Filter -->
        <?php if (!empty($categories)) : ?>
            <div class="mb-10">
                <div class="flex flex-wrap items-center gap-3">
                    <span class="text-gray-600 font-medium"><?php echo esc_html(ai_get_text('blogPage.categoriesLabel', '')); ?></span>
                    <a href="<?php echo home_url('/blog/'); ?>"
                       class="inline-block px-4 py-2 rounded-full text-sm font-medium bg-primary text-white">
                        <?php echo esc_html(ai_get_text('blogPage.allCategories', '')); ?>
                    </a>
                    <?php foreach ($categories as $category) : ?>
                        <a href="<?php echo get_category_link($category->term_id); ?>"
                           class="inline-block px-4 py-2 rounded-full text-sm font-medium bg-gray-100 text-gray-700 hover:bg-gray-200 transition-colors">
                            <?php echo esc_html($category->name); ?>
                            <span class="ml-1 opacity-60">(<?php echo $category->count; ?>)</span>
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Posts Grid -->
        <?php if ($blog_query->have_posts()) : ?>
            <div class="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php while ($blog_query->have_posts()) : $blog_query->the_post();
                    $post_categories = get_the_category();
                    $featured_image = get_the_post_thumbnail_url(get_the_ID(), 'medium_large');
                    $reading_time = ceil(str_word_count(get_the_content()) / 200);
                ?>
                    <article class="bg-white rounded-2xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 group">
                        <!-- Featured Image -->
                        <a href="<?php the_permalink(); ?>" class="block relative overflow-hidden">
                            <?php if ($featured_image) : ?>
                                <img src="<?php echo esc_url($featured_image); ?>"
                                     alt="<?php the_title_attribute(); ?>"
                                     class="w-full h-52 object-cover group-hover:scale-105 transition-transform duration-300">
                            <?php else : ?>
                                <div class="w-full h-52 bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
                                    <svg class="w-16 h-16 text-primary/30" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10a2 2 0 012 2v1m2 13a2 2 0 01-2-2V7m2 13a2 2 0 002-2V9a2 2 0 00-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"></path>
                                    </svg>
                                </div>
                            <?php endif; ?>

                            <!-- Category Badge -->
                            <?php if (!empty($post_categories)) : ?>
                                <span class="absolute top-4 left-4 bg-primary text-white text-xs font-semibold px-3 py-1 rounded-full">
                                    <?php echo esc_html($post_categories[0]->name); ?>
                                </span>
                            <?php endif; ?>
                        </a>

                        <!-- Content -->
                        <div class="p-6">
                            <!-- Meta -->
                            <div class="flex items-center gap-4 text-sm text-gray-500 mb-3">
                                <span><?php echo ai_get_localized_date('j M Y', get_the_time('U')); ?></span>
                                <span class="flex items-center gap-1">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                    <?php echo $reading_time; ?> <?php echo esc_html(ai_get_text('blogPage.minRead', '')); ?>
                                </span>
                            </div>

                            <!-- Title -->
                            <h2 class="text-xl font-bold text-gray-900 mb-3 line-clamp-2 group-hover:text-primary transition-colors">
                                <a href="<?php the_permalink(); ?>">
                                    <?php the_title(); ?>
                                </a>
                            </h2>

                            <!-- Excerpt -->
                            <p class="text-gray-600 text-sm line-clamp-3 mb-4">
                                <?php echo wp_trim_words(get_the_excerpt(), 25); ?>
                            </p>

                            <!-- Read More -->
                            <a href="<?php the_permalink(); ?>"
                               class="inline-flex items-center text-primary font-medium text-sm hover:gap-2 transition-all">
                                <?php echo esc_html(ai_get_text('common.readMore', '')); ?>
                                <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                                </svg>
                            </a>
                        </div>
                    </article>
                <?php endwhile; ?>
            </div>

            <!-- Pagination -->
            <div class="mt-12">
                <?php
                $pagination = paginate_links([
                    'total' => $blog_query->max_num_pages,
                    'current' => $paged,
                    'type' => 'array',
                    'prev_text' => '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg>',
                    'next_text' => '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path></svg>',
                ]);

                if ($pagination) : ?>
                    <nav class="flex justify-center items-center gap-2">
                        <?php foreach ($pagination as $link) :
                            $is_current = strpos($link, 'current') !== false;
                            $is_dots = strpos($link, 'dots') !== false;

                            if ($is_current) {
                                $link = str_replace('page-numbers current', 'w-10 h-10 flex items-center justify-center rounded-full bg-primary text-white font-medium', $link);
                            } elseif ($is_dots) {
                                $link = str_replace('page-numbers dots', 'px-2 text-gray-400', $link);
                            } else {
                                $link = str_replace('page-numbers', 'w-10 h-10 flex items-center justify-center rounded-full bg-gray-100 text-gray-700 hover:bg-gray-200 transition-colors', $link);
                            }

                            echo $link;
                        endforeach; ?>
                    </nav>
                <?php endif;

                wp_reset_postdata();
                ?>
            </div>

        <?php else : ?>
            <!-- No Posts Found -->
            <div class="text-center py-16">
                <div class="w-24 h-24 mx-auto mb-6 bg-gray-100 rounded-full flex items-center justify-center">
                    <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10a2 2 0 012 2v1m2 13a2 2 0 01-2-2V7m2 13a2 2 0 002-2V9a2 2 0 00-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"></path>
                    </svg>
                </div>
                <h2 class="text-2xl font-bold text-gray-900 mb-2"><?php echo esc_html(ai_get_text('blogPage.noPosts', 'No posts found')); ?></h2>
                <p class="text-gray-600 mb-6"><?php echo esc_html(ai_get_text('blogPage.noPostsMessage', 'There are no blog posts yet. Check back soon!')); ?></p>
                <a href="<?php echo home_url('/'); ?>"
                   class="inline-flex items-center px-6 py-3 bg-primary text-white font-medium rounded-lg hover:bg-primary/90 transition-colors">
                    <?php echo esc_html(ai_get_text('blogPage.goHome', 'Go to Homepage')); ?>
                </a>
            </div>
        <?php endif; ?>

    </div>

    <!-- Footer is rendered by footer.php -->
</div>

<?php get_footer(); ?>
