<?php
/**
 * Stats Cards Widget
 *
 * Glassmorphic floating cards with gradient mesh background
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Stats_Cards extends AI_Widget_Base {

    public function get_name() {
        return 'ai_stats_cards';
    }

    public function get_title() {
        return __('Stats Cards', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-number-field';
    }

    public function get_categories() {
        return array('ai-theme-stats');
    }

    public function get_keywords() {
        return array('stats', 'statistics', 'numbers', 'cards', 'glassmorphism', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'emoji',
            array(
                'label'   => __('Emoji', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'value',
            array(
                'label'   => __('Value', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'label',
            array(
                'label'   => __('Label', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'stats',
            array(
                'label'   => __('Stats', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        ?>
        <section
            class="ai-stats ai-stats-cards relative py-24 md:py-32 overflow-hidden"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="stats"
            data-block-variant="stats-cards"
        >
            <!-- Gradient Mesh Background -->
            <div class="absolute inset-0">
                <div class="absolute inset-0 bg-[var(--bg-secondary)]"></div>
                <div class="absolute top-0 right-0 w-[600px] h-[600px] rounded-full blur-3xl -translate-y-1/2 translate-x-1/2 opacity-20" style="background: var(--secondary);"></div>
                <div class="absolute bottom-0 left-0 w-[500px] h-[500px] rounded-full blur-3xl translate-y-1/2 -translate-x-1/2 opacity-20" style="background: var(--primary);"></div>
            </div>

            <div class="relative max-w-6xl mx-auto px-6">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-16">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-4xl md:text-5xl font-bold text-[var(--text-primary)] mb-4">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
                    <?php foreach ($settings['stats'] as $idx => $stat) : ?>
                        <div class="group relative bg-[var(--bg-primary)]/70 backdrop-blur-xl rounded-3xl p-8 border border-[var(--bg-primary)]/50 shadow-lg hover:shadow-xl hover:-translate-y-1 transition-all duration-300">
                            <!-- Gradient Accent -->
                            <div class="absolute inset-0 rounded-3xl bg-gradient-to-br from-[var(--bg-primary)]/80 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>

                            <div class="relative">
                                <!-- Emoji Icon -->
                                <div class="w-14 h-14 rounded-2xl flex items-center justify-center text-2xl mb-5 shadow-lg group-hover:scale-110 transition-transform duration-300" style="background: linear-gradient(to bottom right, var(--primary), var(--secondary)); box-shadow: 0 10px 15px -3px var(--primary-light);">
                                    <?php echo esc_html($stat['emoji'] ?: $this->get_stat_emoji($stat['label'], $idx)); ?>
                                </div>

                                <!-- Number -->
                                <div class="text-4xl md:text-5xl font-bold text-[var(--text-primary)] mb-2 tracking-tight">
                                    <?php echo esc_html($stat['value']); ?>
                                </div>

                                <!-- Label -->
                                <div class="text-[var(--text-muted)] font-medium">
                                    <?php echo esc_html($stat['label']); ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }

    private function get_stat_emoji($label, $index) {
        $label_lower = strtolower($label);

        if (strpos($label_lower, 'user') !== false || strpos($label_lower, 'client') !== false) return '👥';
        if (strpos($label_lower, 'countr') !== false || strpos($label_lower, 'region') !== false) return '🌍';
        if (strpos($label_lower, 'rating') !== false || strpos($label_lower, 'satisf') !== false) return '⭐';
        if (strpos($label_lower, 'project') !== false || strpos($label_lower, 'launch') !== false) return '🚀';
        if (strpos($label_lower, 'year') !== false || strpos($label_lower, 'experi') !== false) return '📅';
        if (strpos($label_lower, 'team') !== false || strpos($label_lower, 'employee') !== false) return '👨‍💻';

        $defaults = array('📊', '✨', '🎯', '💎', '🏆', '⚡');
        return $defaults[$index % count($defaults)];
    }
}
