<?php
/**
 * Products Showcase Featured Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

/**
 * Products Showcase Featured - Bento-style layout with hero product
 */
class Products_Showcase_Featured extends AI_Widget_Base {
    public function get_name() { return 'ai_products_showcase_featured'; }
    public function get_title() { return __('Products Showcase Featured', 'ai-theme'); }
    public function get_icon() { return 'eicon-products'; }
    public function get_categories() { return array('ai-theme-products'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('heading', array(
            'label' => __('Heading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('subheading', array(
            'label' => __('Subheading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('count', array(
            'label' => __('Products Count', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 5,
            'min' => 3,
            'max' => 9,
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $products = wc_get_products(array(
            'limit' => intval($settings['count']),
            'status' => 'publish',
            'featured' => true,
        ));
        
        if (empty($products)) {
            $products = wc_get_products(array(
                'limit' => intval($settings['count']),
                'status' => 'publish',
            ));
        }

        if (empty($products)) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('No products found', 'ai-theme') . '</p>';
            return;
        }

        $hero = array_shift($products);
        $supporting = array_slice($products, 0, 4);
        ?>
        <section class="w-full section-secondary py-16 md:py-24" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="products" data-block-variant="products-showcase-featured">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="flex flex-col md:flex-row md:items-end md:justify-between gap-4 mb-12">
                        <div>
                            <?php if (!empty($settings['subheading'])) : ?>
                                <p class="text-sm font-semibold text-[var(--primary)] tracking-wide uppercase mb-2">
                                    <?php echo esc_html($settings['subheading']); ?>
                                </p>
                            <?php endif; ?>
                            <?php if (!empty($settings['heading'])) : ?>
                                <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)] tracking-tight">
                                    <?php echo esc_html($settings['heading']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Bento Grid -->
                <div class="grid grid-cols-1 lg:grid-cols-12 gap-6">
                    <!-- Hero Product -->
                    <?php if ($hero) : 
                        $hero_image = wp_get_attachment_image_url($hero->get_image_id(), 'large');
                        $hero_rating = $hero->get_average_rating();
                        $hero_categories = wc_get_product_category_list($hero->get_id(), ', ');
                    ?>
                        <div class="lg:col-span-7 lg:row-span-2">
                            <a href="<?php echo esc_url($hero->get_permalink()); ?>" class="group relative block h-full min-h-[500px] lg:min-h-[600px] rounded-3xl overflow-hidden">
                                <!-- Background Image -->
                                <div class="absolute inset-0 bg-slate-900">
                                    <?php if ($hero_image) : ?>
                                        <img src="<?php echo esc_url($hero_image); ?>" alt="<?php echo esc_attr($hero->get_name()); ?>" class="w-full h-full object-cover opacity-80 group-hover:opacity-90 group-hover:scale-105 transition-all duration-700" />
                                    <?php endif; ?>
                                    <div class="absolute inset-0 bg-gradient-to-t from-slate-900 via-slate-900/50 to-transparent"></div>
                                </div>

                                <!-- Featured Badge -->
                                <div class="absolute top-6 left-6 z-10">
                                    <div class="inline-flex items-center gap-2 px-4 py-2 rounded-full font-bold text-sm shadow-lg" style="background: var(--secondary); color: var(--text-primary);">
                                        <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M9.813 15.904L9 18.75l-.813-2.846a4.5 4.5 0 00-3.09-3.09L2.25 12l2.846-.813a4.5 4.5 0 003.09-3.09L9 5.25l.813 2.846a4.5 4.5 0 003.09 3.09L15.75 12l-2.846.813a4.5 4.5 0 00-3.09 3.09z"/></svg>
                                        <?php _e('Featured', 'ai-theme'); ?>
                                    </div>
                                </div>

                                <!-- Content -->
                                <div class="absolute bottom-0 left-0 right-0 p-8 z-10">
                                    <?php if ($hero_categories) : ?>
                                        <p class="text-white/70 text-sm font-medium mb-2">
                                            <?php echo wp_strip_all_tags($hero_categories); ?>
                                        </p>
                                    <?php endif; ?>
                                    <h3 class="text-3xl md:text-4xl font-bold text-white mb-3 group-hover:translate-x-2 transition-transform duration-300">
                                        <?php echo esc_html($hero->get_name()); ?>
                                    </h3>

                                    <?php if ($hero_rating > 0) : ?>
                                        <div class="flex items-center gap-2 mb-4">
                                            <div class="flex">
                                                <?php for ($i = 1; $i <= 5; $i++) : ?>
                                                    <svg class="w-4 h-4 <?php echo $i <= round($hero_rating) ? 'text-amber-400 fill-amber-400' : 'text-white/30'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                                <?php endfor; ?>
                                            </div>
                                            <span class="text-white/70 text-sm">(<?php echo esc_html($hero->get_review_count()); ?> отзывов)</span>
                                        </div>
                                    <?php endif; ?>

                                    <div class="flex items-center justify-between">
                                        <div class="flex items-baseline gap-3">
                                            <span class="text-3xl font-bold text-white"><?php echo $hero->get_price_html(); ?></span>
                                        </div>
                                        <span class="inline-flex items-center gap-2 px-6 py-3 bg-white text-[var(--text-primary)] rounded-full font-bold hover:bg-[var(--secondary)] transition-colors">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                            <?php _e('Buy', 'ai-theme'); ?>
                                        </span>
                                    </div>
                                </div>
                            </a>
                        </div>
                    <?php endif; ?>

                    <!-- Supporting Products -->
                    <?php foreach ($supporting as $product) : 
                        $image = wp_get_attachment_image_url($product->get_image_id(), 'medium');
                        $rating = $product->get_average_rating();
                        $categories = wc_get_product_category_list($product->get_id(), ', ');
                    ?>
                        <div class="lg:col-span-5 lg:col-start-8">
                            <a href="<?php echo esc_url($product->get_permalink()); ?>" class="group relative block h-[280px] rounded-2xl overflow-hidden bg-[var(--bg-primary)] shadow-sm hover:shadow-xl transition-all duration-300">
                                <div class="absolute inset-0 flex">
                                    <!-- Image -->
                                    <div class="w-2/5 bg-[var(--bg-secondary)] overflow-hidden">
                                        <?php if ($image) : ?>
                                            <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500" />
                                        <?php endif; ?>
                                    </div>

                                    <!-- Content -->
                                    <div class="flex-1 p-6 flex flex-col justify-between">
                                        <div>
                                            <?php if ($categories) : ?>
                                                <p class="text-xs text-[var(--text-muted)] uppercase tracking-wide mb-2">
                                                    <?php echo wp_strip_all_tags($categories); ?>
                                                </p>
                                            <?php endif; ?>
                                            <h3 class="text-lg font-bold text-[var(--text-primary)] mb-2 line-clamp-2 group-hover:text-[var(--primary)] transition-colors">
                                                <?php echo esc_html($product->get_name()); ?>
                                            </h3>
                                            <?php if ($rating > 0) : ?>
                                                <div class="flex items-center gap-1">
                                                    <svg class="w-4 h-4 text-amber-400 fill-amber-400" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                                    <span class="text-sm text-[var(--text-secondary)]"><?php echo esc_html($rating); ?></span>
                                                </div>
                                            <?php endif; ?>
                                        </div>

                                        <div class="flex items-center justify-between">
                                            <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                                            <span class="p-2 rounded-full bg-[var(--primary)] text-white hover:opacity-90 transition-colors">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                <?php if ($product->is_on_sale()) : ?>
                                    <div class="absolute top-4 left-4 px-2 py-1 bg-red-500 text-white text-xs font-bold rounded">
                                        <?php _e('Sale', 'ai-theme'); ?>
                                    </div>
                                <?php endif; ?>
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
