<?php
/**
 * Product Split Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Product Split Widget - 50/50 split layout with scrollable images and sticky info
 */
class Product_Split extends AI_Widget_Base {

    public function get_name() {
        return 'ai_product_split';
    }

    public function get_title() {
        return __('Product Split', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-single-product';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('product_id', array(
            'label' => __('Product ID', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
        ));

        $this->add_control('add_to_cart_text', array(
            'label' => __('Add to Cart Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('show_specifications', array(
            'label' => __('Show Specifications', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) return;

        $product_id = !empty($settings['product_id']) ? $settings['product_id'] : get_the_ID();
        $product = wc_get_product($product_id);
        if (!$product) return;

        $image_id = $product->get_image_id();
        $gallery_ids = $product->get_gallery_image_ids();
        $all_images = array_merge(array($image_id), $gallery_ids);
        $all_images = array_filter($all_images);
        
        $widget_id = $this->get_id();
        ?>
        <section class="min-h-screen bg-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="product" data-block-variant="product-split">
            <div class="flex flex-col lg:flex-row">
                <!-- Left - Images (scrollable) -->
                <div class="lg:w-1/2 p-4 lg:p-8">
                    <div class="space-y-4">
                        <?php foreach ($all_images as $img_id) : 
                            $url = wp_get_attachment_image_url($img_id, 'large');
                            $alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                        ?>
                            <div class="aspect-square bg-[var(--bg-secondary)] rounded-[var(--card-radius)] overflow-hidden">
                                <img src="<?php echo esc_url($url); ?>" alt="<?php echo esc_attr($alt ?: $product->get_name()); ?>" class="w-full h-full object-cover" />
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Right - Info (sticky) -->
                <div class="lg:w-1/2 lg:sticky lg:top-0 lg:h-screen lg:overflow-y-auto p-4 lg:p-8">
                    <div class="max-w-lg mx-auto py-8 lg:py-16">
                        <?php 
                        $brand = '';
                        if (taxonomy_exists('pa_brand')) {
                            $brands = wp_get_post_terms($product_id, 'pa_brand');
                            if (!empty($brands) && !is_wp_error($brands)) {
                                $brand = $brands[0]->name;
                            }
                        }
                        if ($brand) : ?>
                            <p class="text-sm text-[var(--primary)] font-medium mb-2"><?php echo esc_html($brand); ?></p>
                        <?php endif; ?>

                        <h1 class="text-3xl lg:text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($product->get_name()); ?></h1>

                        <?php if ($product->get_average_rating()) : ?>
                            <div class="flex items-center gap-2 mb-6">
                                <div class="flex">
                                    <?php for ($i = 1; $i <= 5; $i++) : ?>
                                        <svg class="w-5 h-5 <?php echo $i <= round($product->get_average_rating()) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                    <?php endfor; ?>
                                </div>
                                <span class="text-[var(--text-muted)]"><?php echo esc_html($product->get_average_rating()); ?> (<?php echo esc_html($product->get_review_count()); ?>)</span>
                            </div>
                        <?php endif; ?>

                        <div class="flex items-baseline gap-4 mb-6">
                            <?php if ($product->is_on_sale()) : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo wc_price($product->get_sale_price()); ?></span>
                                <span class="text-xl text-[var(--text-muted)] line-through"><?php echo wc_price($product->get_regular_price()); ?></span>
                            <?php else : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                            <?php endif; ?>
                        </div>

                        <?php if ($product->get_short_description()) : ?>
                            <p class="text-[var(--text-secondary)] mb-6"><?php echo wp_strip_all_tags($product->get_short_description()); ?></p>
                        <?php endif; ?>

                        <?php if ($product->is_in_stock()) : ?>
                            <div class="flex items-center gap-2 text-green-600 mb-6">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                                <span><?php _e('In Stock', 'ai-theme'); ?></span>
                            </div>
                        <?php endif; ?>

                        <!-- Quantity & Add to Cart -->
                        <div class="space-y-4 mb-8">
                            <div class="flex items-center gap-4">
                                <label class="text-sm font-medium text-[var(--text-secondary)]">Qty:</label>
                                <select class="px-4 py-2 border border-[var(--border-color)] rounded-[var(--btn-radius)] bg-[var(--bg-primary)] qty-select-<?php echo esc_attr($widget_id); ?>">
                                    <?php for ($n = 1; $n <= 10; $n++) : ?>
                                        <option value="<?php echo esc_attr($n); ?>"><?php echo esc_html($n); ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>

                            <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-4 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90 flex items-center justify-center gap-2 <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                <?php echo esc_html($settings['add_to_cart_text']); ?>
                            </a>

                            <button class="w-full border border-[var(--border-color)] py-3 rounded-[var(--btn-radius)] font-medium hover:bg-[var(--bg-secondary)] flex items-center justify-center gap-2">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                Save
                            </button>
                        </div>

                        <!-- Benefits -->
                        <div class="space-y-3 py-6 border-t border-[var(--border-color)]">
                            <div class="flex items-center gap-3 text-sm text-[var(--text-secondary)]">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 17H5a2 2 0 01-2-2V5a2 2 0 012-2h14a2 2 0 012 2v10a2 2 0 01-2 2h-3m-1 4l-4 4m0 0l-4-4m4 4V9"/></svg>
                                <span><?php _e('Free Shipping', 'ai-theme'); ?></span>
                            </div>
                            <div class="flex items-center gap-3 text-sm text-[var(--text-secondary)]">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"/></svg>
                                <span><?php _e('Money Back Guarantee', 'ai-theme'); ?></span>
                            </div>
                        </div>

                        <!-- Specifications -->
                        <?php if ($settings['show_specifications'] === 'yes') :
                            $attributes = $product->get_attributes();
                            if (!empty($attributes)) : ?>
                                <div class="py-6 border-t border-[var(--border-color)]">
                                    <h3 class="font-semibold text-[var(--text-primary)] mb-4"><?php _e('Specifications', 'ai-theme'); ?></h3>
                                    <dl class="space-y-2">
                                        <?php foreach (array_slice($attributes, 0, 6) as $attribute) : 
                                            if (!$attribute->get_visible()) continue;
                                            $name = wc_attribute_label($attribute->get_name());
                                            $values = $attribute->is_taxonomy() 
                                                ? implode(', ', wc_get_product_terms($product_id, $attribute->get_name(), array('fields' => 'names')))
                                                : implode(', ', $attribute->get_options());
                                        ?>
                                            <div class="flex justify-between text-sm">
                                                <dt class="text-[var(--text-muted)]"><?php echo esc_html($name); ?></dt>
                                                <dd class="text-[var(--text-primary)]"><?php echo esc_html($values); ?></dd>
                                            </div>
                                        <?php endforeach; ?>
                                    </dl>
                                </div>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </section>
        <?php
    }
}
