<?php
/**
 * Product Minimal Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Product Minimal Widget - Simple product layout without tabs
 */
class Product_Minimal extends AI_Widget_Base {

    public function get_name() {
        return 'ai_product_minimal';
    }

    public function get_title() {
        return __('Product Minimal', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-single-product';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('product_id', array(
            'label' => __('Product ID', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'description' => __('Leave empty for current product', 'ai-theme'),
        ));

        $this->add_control('add_to_cart_text', array(
            'label' => __('Add to Cart Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('show_stock', array(
            'label' => __('Show Stock Status', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) return;

        $product_id = !empty($settings['product_id']) ? $settings['product_id'] : get_the_ID();
        $product = wc_get_product($product_id);
        if (!$product) return;

        $image_id = $product->get_image_id();
        $gallery_ids = $product->get_gallery_image_ids();
        $all_images = array_merge(array($image_id), $gallery_ids);
        $all_images = array_filter($all_images);
        
        $widget_id = $this->get_id();
        ?>
        <section class="py-12 bg-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="product" data-block-variant="product-minimal">
            <div class="max-w-5xl mx-auto px-4">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                    <!-- Images -->
                    <div>
                        <div class="aspect-square bg-[var(--bg-secondary)] rounded-[var(--card-radius)] overflow-hidden mb-4">
                            <?php if (!empty($all_images)) : ?>
                                <img src="<?php echo esc_url(wp_get_attachment_image_url($all_images[0], 'large')); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover gallery-main-<?php echo esc_attr($widget_id); ?>" />
                            <?php endif; ?>
                        </div>
                        <?php if (count($all_images) > 1) : ?>
                            <div class="flex gap-2">
                                <?php foreach (array_slice($all_images, 0, 4) as $idx => $img_id) : 
                                    $thumb_url = wp_get_attachment_image_url($img_id, 'thumbnail');
                                    $large_url = wp_get_attachment_image_url($img_id, 'large');
                                ?>
                                    <button 
                                        class="w-16 h-16 rounded-lg overflow-hidden border-2 gallery-thumb-<?php echo esc_attr($widget_id); ?> <?php echo $idx === 0 ? 'border-[var(--primary)]' : 'border-transparent'; ?>"
                                        data-image="<?php echo esc_url($large_url); ?>"
                                    >
                                        <img src="<?php echo esc_url($thumb_url); ?>" alt="" class="w-full h-full object-cover" />
                                    </button>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Info -->
                    <div class="space-y-4">
                        <?php 
                        $brand = '';
                        if (taxonomy_exists('pa_brand')) {
                            $brands = wp_get_post_terms($product_id, 'pa_brand');
                            if (!empty($brands) && !is_wp_error($brands)) {
                                $brand = $brands[0]->name;
                            }
                        }
                        if ($brand) : ?>
                            <p class="text-sm text-[var(--text-muted)]"><?php echo esc_html($brand); ?></p>
                        <?php endif; ?>

                        <h1 class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($product->get_name()); ?></h1>

                        <?php if ($product->get_average_rating()) : ?>
                            <div class="flex items-center gap-1">
                                <?php for ($i = 1; $i <= 5; $i++) : ?>
                                    <svg class="w-4 h-4 <?php echo $i <= round($product->get_average_rating()) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                <?php endfor; ?>
                                <span class="text-sm text-[var(--text-muted)] ml-2">(<?php echo esc_html($product->get_review_count()); ?>)</span>
                            </div>
                        <?php endif; ?>

                        <div class="flex items-baseline gap-3">
                            <?php if ($product->is_on_sale()) : ?>
                                <span class="text-3xl font-bold text-[var(--text-primary)]"><?php echo wc_price($product->get_sale_price()); ?></span>
                                <span class="text-lg text-[var(--text-muted)] line-through"><?php echo wc_price($product->get_regular_price()); ?></span>
                            <?php else : ?>
                                <span class="text-3xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                            <?php endif; ?>
                        </div>

                        <?php if ($product->get_short_description()) : ?>
                            <p class="text-[var(--text-secondary)]"><?php echo wp_strip_all_tags($product->get_short_description()); ?></p>
                        <?php endif; ?>

                        <?php if ($settings['show_stock'] === 'yes') : ?>
                            <p class="<?php echo $product->is_in_stock() ? 'text-green-600' : 'text-red-500'; ?>">
                                <?php echo $product->is_in_stock() ? __('In Stock', 'ai-theme') : __('Out of Stock', 'ai-theme'); ?>
                            </p>
                        <?php endif; ?>

                        <!-- Quantity -->
                        <div class="flex items-center gap-4">
                            <div class="flex items-center border border-[var(--border-color)] rounded-[var(--btn-radius)]">
                                <button class="p-2 qty-minus-<?php echo esc_attr($widget_id); ?>" type="button">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"/></svg>
                                </button>
                                <input type="number" class="qty-input-<?php echo esc_attr($widget_id); ?> w-12 text-center font-medium border-0 bg-transparent" value="1" min="1" />
                                <button class="p-2 qty-plus-<?php echo esc_attr($widget_id); ?>" type="button">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                                </button>
                            </div>
                        </div>

                        <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="block w-full bg-[var(--primary)] text-white py-3 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90 flex items-center justify-center gap-2 <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                            <?php echo esc_html($settings['add_to_cart_text']); ?>
                        </a>
                    </div>
                </div>
            </div>
        </section>

        <script>
        (function() {
            // Gallery
            const mainImg = document.querySelector('.gallery-main-<?php echo esc_attr($widget_id); ?>');
            const thumbs = document.querySelectorAll('.gallery-thumb-<?php echo esc_attr($widget_id); ?>');
            thumbs.forEach(thumb => {
                thumb.addEventListener('click', function() {
                    if (mainImg) mainImg.src = this.dataset.image;
                    thumbs.forEach(t => {
                        t.classList.remove('border-[var(--primary)]');
                        t.classList.add('border-transparent');
                    });
                    this.classList.remove('border-transparent');
                    this.classList.add('border-[var(--primary)]');
                });
            });

            // Quantity
            const qtyInput = document.querySelector('.qty-input-<?php echo esc_attr($widget_id); ?>');
            const qtyMinus = document.querySelector('.qty-minus-<?php echo esc_attr($widget_id); ?>');
            const qtyPlus = document.querySelector('.qty-plus-<?php echo esc_attr($widget_id); ?>');
            if (qtyMinus && qtyInput) {
                qtyMinus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    if (val > 1) qtyInput.value = val - 1;
                });
            }
            if (qtyPlus && qtyInput) {
                qtyPlus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    qtyInput.value = val + 1;
                });
            }
        })();
        </script>
        <?php
    }
}
