<?php
/**
 * Hero Minimal Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Hero Minimal Widget
 */
class Hero_Minimal extends AI_Widget_Base {

    /**
     * Get widget name
     *
     * @return string
     */
    public function get_name() {
        return 'ai_hero_minimal';
    }

    /**
     * Get widget title
     *
     * @return string
     */
    public function get_title() {
        return __('Hero Minimal', 'ai-theme');
    }

    /**
     * Get widget icon
     *
     * @return string
     */
    public function get_icon() {
        return 'eicon-banner';
    }

    /**
     * Get widget categories
     *
     * @return array
     */
    public function get_categories() {
        return array('ai-theme-hero', 'ai-theme-blocks');
    }

    /**
     * Get widget keywords
     *
     * @return array
     */
    public function get_keywords() {
        return array('hero', 'banner', 'header', 'minimal', 'ai', 'landing');
    }

    /**
     * Register controls
     */
    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'       => __('Heading', 'ai-theme'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'placeholder' => __('Enter heading', 'ai-theme'),
                'label_block' => true,
                'dynamic'     => array('active' => true),
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'       => __('Subheading', 'ai-theme'),
                'type'        => Controls_Manager::TEXTAREA,
                'default'     => '',
                'placeholder' => __('Enter subheading', 'ai-theme'),
                'rows'        => 3,
                'dynamic'     => array('active' => true),
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'       => __('Button Text', 'ai-theme'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'placeholder' => __('Enter button text', 'ai-theme'),
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label'       => __('Button Link', 'ai-theme'),
                'type'        => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'ai-theme'),
                'default'     => array(
                    'url'         => '#',
                    'is_external' => false,
                    'nofollow'    => false,
                ),
                'dynamic'     => array('active' => true),
            )
        );

        $this->add_control(
            'secondary_cta_text',
            array(
                'label'       => __('Secondary Button Text', 'ai-theme'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'placeholder' => __('Learn More', 'ai-theme'),
            )
        );

        $this->add_control(
            'secondary_cta_link',
            array(
                'label'       => __('Secondary Button Link', 'ai-theme'),
                'type'        => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'ai-theme'),
                'condition'   => array(
                    'secondary_cta_text!' => '',
                ),
            )
        );

        $this->end_controls_section();

        // Background Section
        $this->start_controls_section(
            'background_section',
            array(
                'label' => __('Background', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'background_type',
            array(
                'label'   => __('Background Type', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'gradient',
                'options' => array(
                    'gradient' => __('Gradient', 'ai-theme'),
                    'image'    => __('Image', 'ai-theme'),
                    'color'    => __('Solid Color', 'ai-theme'),
                ),
            )
        );

        $this->add_control(
            'background_image',
            array(
                'label'     => __('Background Image', 'ai-theme'),
                'type'      => Controls_Manager::MEDIA,
                'default'   => array(
                    'url' => '',
                ),
                'condition' => array(
                    'background_type' => 'image',
                ),
            )
        );

        $this->add_control(
            'overlay_opacity',
            array(
                'label'     => __('Overlay Opacity', 'ai-theme'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => array(
                    'px' => array(
                        'min'  => 0,
                        'max'  => 1,
                        'step' => 0.1,
                    ),
                ),
                'default'   => array(
                    'size' => 0.65,
                ),
                'condition' => array(
                    'background_type' => 'image',
                ),
            )
        );

        $this->add_control(
            'background_color',
            array(
                'label'     => __('Background Color', 'ai-theme'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'condition' => array(
                    'background_type' => 'color',
                ),
            )
        );

        $this->end_controls_section();

        // Layout Section
        $this->start_controls_section(
            'layout_section',
            array(
                'label' => __('Layout', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'height',
            array(
                'label'   => __('Height', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => '500px',
                'options' => array(
                    '400px'  => __('Small (400px)', 'ai-theme'),
                    '500px'  => __('Medium (500px)', 'ai-theme'),
                    '600px'  => __('Large (600px)', 'ai-theme'),
                    '100vh'  => __('Full Screen', 'ai-theme'),
                    'custom' => __('Custom', 'ai-theme'),
                ),
            )
        );

        $this->add_control(
            'custom_height',
            array(
                'label'     => __('Custom Height', 'ai-theme'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => array(
                    'px' => array(
                        'min' => 200,
                        'max' => 1000,
                    ),
                    'vh' => array(
                        'min' => 20,
                        'max' => 100,
                    ),
                ),
                'default'   => array(
                    'size' => 500,
                    'unit' => 'px',
                ),
                'size_units' => array('px', 'vh'),
                'condition' => array(
                    'height' => 'custom',
                ),
            )
        );

        $this->add_control(
            'content_alignment',
            array(
                'label'   => __('Content Alignment', 'ai-theme'),
                'type'    => Controls_Manager::CHOOSE,
                'options' => array(
                    'left'   => array(
                        'title' => __('Left', 'ai-theme'),
                        'icon'  => 'eicon-text-align-left',
                    ),
                    'center' => array(
                        'title' => __('Center', 'ai-theme'),
                        'icon'  => 'eicon-text-align-center',
                    ),
                    'right'  => array(
                        'title' => __('Right', 'ai-theme'),
                        'icon'  => 'eicon-text-align-right',
                    ),
                ),
                'default' => 'center',
            )
        );

        $this->add_theme_style_control();

        $this->end_controls_section();

        // Style Section - Heading
        $this->start_controls_section(
            'style_heading',
            array(
                'label' => __('Heading Style', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_STYLE,
            )
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            array(
                'name'     => 'heading_typography',
                'selector' => '{{WRAPPER}} .ai-hero-heading',
            )
        );

        $this->add_control(
            'heading_color',
            array(
                'label'     => __('Heading Color', 'ai-theme'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#111827',  // gray-900 - dark text on light bg
                'selectors' => array(
                    '{{WRAPPER}} .ai-hero-heading' => 'color: {{VALUE}};',
                ),
            )
        );

        $this->end_controls_section();

        // Style Section - Subheading
        $this->start_controls_section(
            'style_subheading',
            array(
                'label' => __('Subheading Style', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_STYLE,
            )
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            array(
                'name'     => 'subheading_typography',
                'selector' => '{{WRAPPER}} .ai-hero-subheading',
            )
        );

        $this->add_control(
            'subheading_color',
            array(
                'label'     => __('Subheading Color', 'ai-theme'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#4b5563',  // gray-600 - subtle dark text
                'selectors' => array(
                    '{{WRAPPER}} .ai-hero-subheading' => 'color: {{VALUE}};',
                ),
            )
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        // Get theme
        $theme = $settings['theme_style'] ?? 'inherit';
        if ($theme === 'inherit') {
            $theme = get_option('ai_theme_style', 'modern');
        }
        ?>
        <div data-theme="<?php echo esc_attr($theme); ?>" class="ai-theme-block">
            <section
                class="ai-hero ai-hero-minimal relative min-h-[80vh] flex items-center justify-center section-light"
                data-block-type="hero"
                data-block-variant="hero-minimal"
            >
                <!-- Subtle background pattern -->
                <div class="absolute inset-0 opacity-5" style="background-image: radial-gradient(circle at 2px 2px, currentColor 1px, transparent 0); background-size: 40px 40px;"></div>

                <div class="relative max-w-4xl mx-auto px-6 md:px-8 text-center space-y-8 py-20">
                    <?php if (!empty($settings['heading'])) : ?>
                        <h1 class="ai-hero-heading text-5xl md:text-6xl lg:text-7xl font-light tracking-tight text-[var(--text-primary)]">
                            <?php echo esc_html($settings['heading']); ?>
                        </h1>
                    <?php endif; ?>

                    <!-- Divider line -->
                    <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                        <div class="flex justify-center">
                            <div class="w-20 h-px bg-[var(--border-color)] opacity-50"></div>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($settings['subheading'])) : ?>
                        <p class="ai-hero-subheading text-lg md:text-xl lg:text-2xl font-light text-[var(--text-secondary)] max-w-2xl mx-auto opacity-80">
                            <?php echo esc_html($settings['subheading']); ?>
                        </p>
                    <?php endif; ?>

                    <?php if (!empty($settings['cta_text'])) : ?>
                        <div class="pt-4">
                            <a
                                href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>"
                                class="inline-block px-8 py-3 md:px-10 md:py-4 rounded-full border-2 border-[var(--border-color)] text-[var(--text-secondary)] hover:border-[var(--primary)] hover:text-[var(--primary)] hover:shadow-lg transition-all duration-300"
                                <?php echo !empty($settings['cta_link']['is_external']) ? 'target="_blank"' : ''; ?>
                                <?php echo !empty($settings['cta_link']['nofollow']) ? 'rel="nofollow"' : ''; ?>
                            >
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </section>
        </div>
        <?php
    }

    /**
     * Render widget output in editor
     */
    protected function content_template() {
        ?>
        <#
        var theme = settings.theme_style || 'modern';
        if (theme === 'inherit') theme = 'modern';

        var height = settings.height;
        if (height === 'custom') {
            height = settings.custom_height.size + settings.custom_height.unit;
        }

        var bgStyle = '';
        var sectionClass = 'section-secondary';

        if (settings.background_type === 'image' && settings.background_image.url) {
            var opacity = settings.overlay_opacity.size || 0.65;
            bgStyle = 'background-image: linear-gradient(rgba(0,0,0,' + opacity + '), rgba(0,0,0,' + opacity + ')), url(' + settings.background_image.url + '); background-size: cover; background-position: center;';
            sectionClass = '';
        } else if (settings.background_type === 'color' && settings.background_color) {
            bgStyle = 'background-color: ' + settings.background_color + ';';
            sectionClass = '';
        }

        var alignmentClass = 'text-center justify-center';
        if (settings.content_alignment === 'left') {
            alignmentClass = 'text-left justify-start';
        } else if (settings.content_alignment === 'right') {
            alignmentClass = 'text-right justify-end';
        }
        #>
        <div data-theme="{{ theme }}" class="ai-theme-block">
            <section
                class="{{ sectionClass }} relative flex items-center {{ alignmentClass }}"
                data-block-type="hero"
                data-block-variant="hero-minimal"
                style="height: {{ height }}; {{{ bgStyle }}}"
            >
                <div class="ai-container <# if (settings.content_alignment === 'center') { #>text-center<# } #>">
                    <# if (settings.heading) { #>
                        <h1 class="ai-hero-heading heading-primary text-5xl md:text-6xl font-bold mb-4 text-[var(--text-primary)]">
                            {{{ settings.heading }}}
                        </h1>
                    <# } #>

                    <# if (settings.subheading) { #>
                        <p class="ai-hero-subheading text-secondary text-xl md:text-2xl mb-8 opacity-90 text-[var(--text-secondary)]">
                            {{{ settings.subheading }}}
                        </p>
                    <# } #>

                    <div class="flex gap-4 <# if (settings.content_alignment === 'center') { #>justify-center<# } else if (settings.content_alignment === 'right') { #>justify-end<# } else { #>justify-start<# } #>">
                        <# if (settings.cta_text) { #>
                            <a href="{{ settings.cta_link.url }}" class="btn-primary px-8 py-4 shadow-lg text-lg font-medium rounded-lg">
                                {{{ settings.cta_text }}}
                            </a>
                        <# } #>

                        <# if (settings.secondary_cta_text) { #>
                            <a href="{{ settings.secondary_cta_link.url }}" class="btn-outline px-8 py-4 text-lg font-medium rounded-lg border-[var(--border-color)] text-[var(--text-secondary)]">
                                {{{ settings.secondary_cta_text }}}
                            </a>
                        <# } #>
                    </div>
                </div>
            </section>
        </div>
        <?php
    }
}
