<?php
/**
 * Hero Image Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Hero Image Widget - Fullscreen image with overlay
 */
class Hero_Image extends AI_Widget_Base {

    public function get_name() {
        return 'ai_hero_image';
    }

    public function get_title() {
        return __('Hero Image', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-banner';
    }

    public function get_categories() {
        return array('ai-theme-hero', 'ai-theme-blocks');
    }

    public function get_keywords() {
        return array('hero', 'banner', 'image', 'fullscreen', 'ai', 'landing');
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'       => __('Heading', 'ai-theme'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'label_block' => true,
                'dynamic'     => array('active' => true),
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'       => __('Subheading', 'ai-theme'),
                'type'        => Controls_Manager::TEXTAREA,
                'default'     => '',
                'rows'        => 3,
                'dynamic'     => array('active' => true),
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label'   => __('Button Link', 'ai-theme'),
                'type'    => Controls_Manager::URL,
                'default' => array('url' => '#'),
            )
        );

        $this->end_controls_section();

        // Background Section
        $this->start_controls_section(
            'background_section',
            array(
                'label' => __('Background', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'background_image',
            array(
                'label'   => __('Background Image', 'ai-theme'),
                'type'    => Controls_Manager::MEDIA,
                'default' => array('url' => ''),
            )
        );

        $this->end_controls_section();

        // Layout
        $this->start_controls_section(
            'layout_section',
            array(
                'label' => __('Layout', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_theme_style_control();

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $theme = $settings['theme_style'] ?? 'inherit';
        if ($theme === 'inherit') {
            $theme = get_option('ai_theme_style', 'modern');
        }

        $bg_url = !empty($settings['background_image']['url']) ? esc_url($settings['background_image']['url']) : '';
        ?>
        <div data-theme="<?php echo esc_attr($theme); ?>" class="ai-theme-block">
            <section
                class="ai-hero ai-hero-image relative min-h-screen flex items-center justify-center overflow-hidden"
                data-block-type="hero"
                data-block-variant="hero-image"
            >
                <!-- Background Image -->
                <?php if ($bg_url) : ?>
                    <div class="absolute inset-0 bg-cover bg-center bg-no-repeat" style="background-image: url(<?php echo $bg_url; ?>);"></div>
                <?php else : ?>
                    <div class="absolute inset-0" style="background: linear-gradient(to bottom right, var(--primary), var(--secondary));"></div>
                <?php endif; ?>

                <!-- Dark overlay -->
                <div class="absolute inset-0 bg-gradient-to-t from-black/80 via-black/50 to-black/30"></div>

                <!-- Content -->
                <div class="relative z-10 max-w-4xl mx-auto px-6 md:px-8 text-center">
                    <?php if (!empty($settings['heading'])) : ?>
                        <h1 class="ai-hero-heading text-4xl md:text-5xl lg:text-6xl xl:text-7xl font-bold text-white leading-tight mb-6">
                            <?php echo esc_html($settings['heading']); ?>
                        </h1>
                    <?php endif; ?>

                    <?php if (!empty($settings['subheading'])) : ?>
                        <p class="ai-hero-subheading text-lg md:text-xl lg:text-2xl text-white/80 max-w-2xl mx-auto mb-8">
                            <?php echo esc_html($settings['subheading']); ?>
                        </p>
                    <?php endif; ?>

                    <?php if (!empty($settings['cta_text'])) : ?>
                        <div class="flex flex-col sm:flex-row items-center justify-center gap-4">
                            <a
                                href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>"
                                class="px-8 py-4 bg-[var(--bg-primary)] text-[var(--text-primary)] font-semibold rounded-lg hover:opacity-90 transition-all duration-300 shadow-xl hover:shadow-2xl"
                                <?php echo !empty($settings['cta_link']['is_external']) ? 'target="_blank"' : ''; ?>
                            >
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Scroll indicator -->
                <div class="absolute bottom-8 left-1/2 -translate-x-1/2 animate-bounce">
                    <svg class="w-6 h-6 text-white/60" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 14l-7 7m0 0l-7-7m7 7V3" />
                    </svg>
                </div>
            </section>
        </div>
        <?php
    }
}
