<?php
/**
 * Footer Corporate Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Footer Corporate Widget - Professional corporate footer
 */
class Footer_Corporate extends AI_Widget_Base {

    public function get_name() {
        return 'ai_footer_corporate';
    }

    public function get_title() {
        return __('Footer Corporate', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-footer';
    }

    public function get_categories() {
        return array('ai-theme-footer');
    }

    public function get_keywords() {
        return array('footer', 'corporate', 'business', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'logo_text',
            array(
                'label'   => __('Logo Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'logo_image',
            array(
                'label' => __('Logo Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $this->add_control(
            'description',
            array(
                'label'   => __('Description', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'phone',
            array(
                'label'   => __('Phone', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'email',
            array(
                'label'   => __('Email', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'address',
            array(
                'label'   => __('Address', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'working_hours',
            array(
                'label'   => __('Working Hours', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'copyright',
            array(
                'label'   => __('Copyright Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->end_controls_section();

        // Footer Columns
        $this->start_controls_section(
            'columns_section',
            array(
                'label' => __('Link Columns', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $columns_repeater = new Repeater();

        $columns_repeater->add_control(
            'column_title',
            array(
                'label'   => __('Column Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $columns_repeater->add_control(
            'column_links',
            array(
                'label'   => __('Links (one per line: Label|URL)', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'footer_columns',
            array(
                'label'   => __('Columns', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $columns_repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();

        // Social Links
        $this->start_controls_section(
            'social_section',
            array(
                'label' => __('Social Links', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $social_repeater = new Repeater();

        $social_repeater->add_control(
            'platform',
            array(
                'label'   => __('Platform', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'linkedin',
                'options' => array(
                    'facebook'  => 'Facebook',
                    'twitter'   => 'Twitter/X',
                    'instagram' => 'Instagram',
                    'linkedin'  => 'LinkedIn',
                    'youtube'   => 'YouTube',
                ),
            )
        );

        $social_repeater->add_control(
            'link',
            array(
                'label' => __('Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'social_links',
            array(
                'label'  => __('Social Links', 'ai-theme'),
                'type'   => Controls_Manager::REPEATER,
                'fields' => $social_repeater->get_controls(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $logo_url = !empty($settings['logo_image']['url']) ? $settings['logo_image']['url'] : '';
        ?>
        <footer
            class="ai-footer bg-[var(--bg-dark,#1e293b)] text-[var(--text-on-dark)]"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="footer"
            data-block-variant="footer-corporate"
        >
            <!-- Main Content -->
            <div class="max-w-7xl mx-auto px-6 py-16">
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-12">
                    <!-- Brand & About -->
                    <div class="lg:col-span-2">
                        <a href="<?php echo esc_url(home_url('/')); ?>" class="inline-block mb-6">
                            <?php if ($logo_url) : ?>
                                <img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr($settings['logo_text']); ?>" class="h-10 object-contain brightness-0 invert" />
                            <?php else : ?>
                                <h3 class="text-2xl font-bold"><?php echo esc_html($settings['logo_text']); ?></h3>
                            <?php endif; ?>
                        </a>
                        <p class="text-[var(--text-muted)] mb-6 leading-relaxed">
                            <?php echo esc_html($settings['description']); ?>
                        </p>

                        <!-- Social Links -->
                        <?php if (!empty($settings['social_links'])) : ?>
                            <div>
                                <div class="flex gap-2">
                                    <?php foreach ($settings['social_links'] as $social) : ?>
                                        <a
                                            href="<?php echo esc_url($social['link']['url'] ?? '#'); ?>"
                                            target="_blank"
                                            rel="noopener noreferrer"
                                            class="w-9 h-9 bg-white/10 rounded flex items-center justify-center hover:bg-[var(--primary-on-dark,var(--primary))] transition-colors"
                                            aria-label="<?php echo esc_attr($social['platform']); ?>"
                                        >
                                            <?php echo $this->get_social_icon_svg($social['platform']); ?>
                                        </a>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Link Columns (фильтруем Legal колонку - она показывается внизу) -->
                    <?php 
                    foreach ($settings['footer_columns'] as $column) : 
                        // Пропускаем колонку Legal - она дублируется внизу
                        $title_lower = strtolower($column['column_title'] ?? '');
                        $is_legal_column = (
                            strpos($title_lower, 'legal') !== false || 
                            strpos($title_lower, 'юридич') !== false || 
                            strpos($title_lower, 'правов') !== false
                        );
                        
                        // Проверяем, содержит ли колонка только legal ссылки
                        if ($is_legal_column) {
                            $links = explode("\n", $column['column_links']);
                            $has_only_legal = true;
                            foreach ($links as $link_line) {
                                $parts = explode('|', trim($link_line));
                                if (count($parts) >= 2) {
                                    $url = strtolower(trim($parts[1]));
                                    if (strpos($url, 'privacy') === false && strpos($url, 'cookies') === false && strpos($url, 'terms') === false) {
                                        $has_only_legal = false;
                                        break;
                                    }
                                }
                            }
                            if ($has_only_legal) continue; // Пропускаем эту колонку
                        }
                    ?>
                        <div>
                            <h4 class="text-sm font-semibold text-[var(--text-on-dark)] uppercase tracking-wider mb-4"><?php echo esc_html($column['column_title']); ?></h4>
                            <ul class="space-y-3 text-sm">
                                <?php 
                                $links = explode("\n", $column['column_links']);
                                foreach ($links as $link_line) :
                                    $parts = explode('|', trim($link_line));
                                    if (count($parts) >= 2) :
                                        $label = trim($parts[0]);
                                        $url = trim($parts[1]);
                                ?>
                                    <li>
                                        <a href="<?php echo esc_url($url); ?>" class="text-[var(--text-muted)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                            <?php echo esc_html($label); ?>
                                        </a>
                                    </li>
                                <?php 
                                    endif;
                                endforeach; 
                                ?>
                            </ul>
                        </div>
                    <?php endforeach; ?>

                    <!-- Contact Info -->
                    <div>
                        <h4 class="text-sm font-semibold text-[var(--text-on-dark)] uppercase tracking-wider mb-4"><?php echo esc_html(ai_get_text('navigation.labels.contact', '')); ?></h4>
                        <div class="space-y-4 text-sm">
                            <?php if (!empty($settings['phone'])) : ?>
                                <a href="tel:<?php echo esc_attr($settings['phone']); ?>" class="flex items-center gap-3 text-[var(--text-muted)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                    <svg class="w-4 h-4 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"></path></svg>
                                    <span><?php echo esc_html($settings['phone']); ?></span>
                                </a>
                            <?php endif; ?>
                            <?php if (!empty($settings['email'])) : ?>
                                <a href="mailto:<?php echo esc_attr($settings['email']); ?>" class="flex items-center gap-3 text-[var(--text-muted)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                    <svg class="w-4 h-4 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path></svg>
                                    <span><?php echo esc_html($settings['email']); ?></span>
                                </a>
                            <?php endif; ?>
                            <?php if (!empty($settings['address'])) : ?>
                                <div class="flex items-start gap-3 text-[var(--text-muted)]">
                                    <svg class="w-4 h-4 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                                    <span><?php echo nl2br(esc_html($settings['address'])); ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($settings['working_hours'])) : ?>
                                <div class="flex items-start gap-3 text-[var(--text-muted)] pt-2">
                                    <svg class="w-4 h-4 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                    <div>
                                        <p><?php echo esc_html($settings['working_hours']); ?></p>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Legal Bar -->
            <div class="border-t border-white/10">
                <div class="max-w-7xl mx-auto px-6 py-6">
                    <div class="flex flex-col lg:flex-row justify-between items-center gap-4">
                        <p class="text-sm text-[var(--text-muted)]">
                            <?php echo esc_html($settings['copyright']); ?>
                        </p>
                        <div class="flex flex-wrap justify-center gap-6 text-sm">
                            <a href="<?php echo esc_url(home_url('/privacy-policy')); ?>" class="text-[var(--text-muted)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                <?php echo esc_html(ai_get_text('navigation.labels.privacy', 'Privacy Policy')); ?>
                            </a>
                            <a href="<?php echo esc_url(home_url('/cookies')); ?>" class="text-[var(--text-muted)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                <?php echo esc_html(ai_get_text('navigation.labels.cookies', 'Cookie Policy')); ?>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </footer>
        <?php
    }

    private function get_social_icon_svg($platform) {
        $icons = array(
            'facebook'  => '<svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>',
            'twitter'   => '<svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>',
            'instagram' => '<svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"/></svg>',
            'linkedin'  => '<svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/></svg>',
            'youtube'   => '<svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/></svg>',
        );
        return $icons[$platform] ?? '';
    }
}
