<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

class Filters_Sidebar extends AI_Widget_Base {
    public function get_name() { return 'ai_filters_sidebar'; }
    public function get_title() { return __('Filters Sidebar', 'ai-theme'); }
    public function get_icon() { return 'eicon-filter'; }
    public function get_categories() { return array('ai-theme-filters'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);
        $this->add_control('title', ['label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => '']);
        $this->add_control('description', ['label' => __('Description', 'ai-theme'), 'type' => Controls_Manager::TEXTAREA, 'default' => '']);
        $this->add_control('show_categories', ['label' => __('Show Categories', 'ai-theme'), 'type' => Controls_Manager::SWITCHER, 'default' => 'yes']);
        $this->add_control('show_price', ['label' => __('Show Price Filter', 'ai-theme'), 'type' => Controls_Manager::SWITCHER, 'default' => 'yes']);
        $this->add_control('show_attributes', ['label' => __('Show Product Attributes', 'ai-theme'), 'type' => Controls_Manager::SWITCHER, 'default' => 'yes']);
        $this->add_control('show_reset', ['label' => __('Show Reset Button', 'ai-theme'), 'type' => Controls_Manager::SWITCHER, 'default' => 'yes']);
        $this->add_control('reset_text', ['label' => __('Reset Button Text', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => '']);
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $widget_id = $this->get_id();
        
        // Get WooCommerce data
        $categories = [];
        $attributes = [];
        $price_range = ['min' => 0, 'max' => 100000];
        
        if (class_exists('WooCommerce')) {
            // Categories
            if ($settings['show_categories'] === 'yes') {
                $categories = get_terms([
                    'taxonomy' => 'product_cat',
                    'hide_empty' => true,
                    'number' => 10,
                ]);
            }
            
            // Price range
            if ($settings['show_price'] === 'yes') {
                global $wpdb;
                $prices = $wpdb->get_row("
                    SELECT MIN(CAST(meta_value AS DECIMAL(10,2))) as min_price, 
                           MAX(CAST(meta_value AS DECIMAL(10,2))) as max_price 
                    FROM {$wpdb->postmeta} 
                    WHERE meta_key = '_price' AND meta_value > 0
                ");
                if ($prices) {
                    $price_range['min'] = floor($prices->min_price / 100) * 100;
                    $price_range['max'] = ceil($prices->max_price / 100) * 100;
                }
            }
            
            // Product attributes
            if ($settings['show_attributes'] === 'yes') {
                $attribute_taxonomies = wc_get_attribute_taxonomies();
                foreach ($attribute_taxonomies as $tax) {
                    $terms = get_terms([
                        'taxonomy' => 'pa_' . $tax->attribute_name,
                        'hide_empty' => true,
                    ]);
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $attributes[] = [
                            'name' => $tax->attribute_label,
                            'slug' => $tax->attribute_name,
                            'terms' => $terms,
                        ];
                    }
                }
            }
        }
        ?>
        <div class="w-full bg-[var(--bg-primary)] rounded-xl shadow-lg border border-[var(--border-color)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="filters" data-block-variant="filters-sidebar">
            <!-- Header -->
            <div class="bg-gradient-to-br from-blue-50 to-indigo-50 p-6 rounded-t-xl border-b border-[var(--border-color)]">
                <div class="flex items-center gap-2 mb-2">
                    <div class="p-2 bg-[var(--bg-primary)] rounded-lg shadow-sm">
                        <svg class="w-5 h-5 text-[var(--primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4"></path>
                        </svg>
                    </div>
                    <h3 class="text-xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['title']); ?></h3>
                    <span class="active-count hidden inline-flex items-center justify-center min-w-[24px] h-6 px-2 bg-[var(--primary)] text-white text-xs font-bold rounded-full"></span>
                </div>
                <?php if (!empty($settings['description'])) : ?>
                <p class="text-sm text-[var(--text-secondary)]"><?php echo esc_html($settings['description']); ?></p>
                <?php endif; ?>
                
                <?php if ($settings['show_reset'] === 'yes') : ?>
                <button type="button" class="reset-filters hidden w-full mt-3 px-4 py-2 bg-[var(--bg-primary)] text-[var(--primary)] border border-[var(--primary-light,#dbeafe)] rounded-lg hover:bg-[var(--primary-light,#dbeafe)] font-medium items-center justify-center gap-2 transition-all">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                    <?php echo esc_html($settings['reset_text']); ?>
                </button>
                <?php endif; ?>
            </div>

            <!-- Filter Groups -->
            <div class="divide-y divide-[var(--border-color)]">
                <?php if (!empty($categories)) : ?>
                <!-- Categories -->
                <div class="filter-group p-5" data-filter="category">
                    <button type="button" class="w-full flex items-center justify-between mb-4 group filter-toggle">
                        <span class="font-semibold text-[var(--text-primary)]"><?php _e('Categories', 'ai-theme'); ?></span>
                        <svg class="w-4 h-4 text-[var(--text-muted)] transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 15l7-7 7 7"></path>
                        </svg>
                    </button>
                    <div class="filter-content space-y-2.5">
                        <?php foreach ($categories as $cat) : ?>
                        <label class="flex items-center gap-3 p-2.5 rounded-lg cursor-pointer transition-all hover:bg-[var(--bg-secondary)] border border-transparent">
                            <input type="checkbox" name="filter_category[]" value="<?php echo esc_attr($cat->slug); ?>" class="w-5 h-5 rounded border-[var(--border-color)] text-[var(--primary)] focus:ring-2 focus:ring-[var(--primary)]">
                            <span class="flex-1 text-sm font-medium text-[var(--text-primary)]"><?php echo esc_html($cat->name); ?></span>
                            <span class="text-xs px-2 py-0.5 rounded-full bg-[var(--bg-secondary)] text-[var(--text-secondary)]"><?php echo $cat->count; ?></span>
                        </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($settings['show_price'] === 'yes') : ?>
                <!-- Price Range -->
                <div class="filter-group p-5" data-filter="price">
                    <button type="button" class="w-full flex items-center justify-between mb-4 group filter-toggle">
                        <span class="font-semibold text-[var(--text-primary)]"><?php _e('Price', 'ai-theme'); ?></span>
                        <svg class="w-4 h-4 text-[var(--text-muted)] transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 15l7-7 7 7"></path>
                        </svg>
                    </button>
                    <div class="filter-content space-y-4">
                        <div class="flex items-center justify-between text-sm text-[var(--text-secondary)]">
                            <span class="price-min-label"><?php echo wc_price($price_range['min']); ?></span>
                            <span class="price-max-label"><?php echo wc_price($price_range['max']); ?></span>
                        </div>
                        <div class="space-y-2">
                            <input type="range" name="filter_price_min" min="<?php echo $price_range['min']; ?>" max="<?php echo $price_range['max']; ?>" value="<?php echo $price_range['min']; ?>" step="100" class="price-slider w-full h-2 bg-[var(--bg-secondary)] rounded-lg appearance-none cursor-pointer" style="accent-color: var(--primary);">
                            <input type="range" name="filter_price_max" min="<?php echo $price_range['min']; ?>" max="<?php echo $price_range['max']; ?>" value="<?php echo $price_range['max']; ?>" step="100" class="price-slider w-full h-2 bg-[var(--bg-secondary)] rounded-lg appearance-none cursor-pointer" style="accent-color: var(--primary);">
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <?php foreach ($attributes as $attr) : ?>
                <!-- Attribute: <?php echo esc_html($attr['name']); ?> -->
                <div class="filter-group p-5" data-filter="pa_<?php echo esc_attr($attr['slug']); ?>">
                    <button type="button" class="w-full flex items-center justify-between mb-4 group filter-toggle">
                        <span class="font-semibold text-[var(--text-primary)]"><?php echo esc_html($attr['name']); ?></span>
                        <svg class="w-4 h-4 text-[var(--text-muted)] transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 15l7-7 7 7"></path>
                        </svg>
                    </button>
                    <div class="filter-content space-y-2.5">
                        <?php foreach ($attr['terms'] as $term) : ?>
                        <label class="flex items-center gap-3 p-2.5 rounded-lg cursor-pointer transition-all hover:bg-[var(--bg-secondary)] border border-transparent">
                            <input type="checkbox" name="filter_<?php echo esc_attr($attr['slug']); ?>[]" value="<?php echo esc_attr($term->slug); ?>" class="w-5 h-5 rounded border-[var(--border-color)] text-[var(--primary)] focus:ring-2 focus:ring-[var(--primary)]">
                            <span class="flex-1 text-sm font-medium text-[var(--text-primary)]"><?php echo esc_html($term->name); ?></span>
                            <span class="text-xs px-2 py-0.5 rounded-full bg-[var(--bg-secondary)] text-[var(--text-secondary)]"><?php echo $term->count; ?></span>
                        </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>

        <script>
        (function() {
            const widget = document.querySelector('[data-block-variant="filters-sidebar"]');
            if (!widget) return;

            // Toggle filter sections
            widget.querySelectorAll('.filter-toggle').forEach(btn => {
                btn.addEventListener('click', () => {
                    const content = btn.nextElementSibling;
                    const icon = btn.querySelector('svg');
                    content.classList.toggle('hidden');
                    icon.style.transform = content.classList.contains('hidden') ? 'rotate(180deg)' : '';
                });
            });

            // Update active count
            function updateActiveCount() {
                const checked = widget.querySelectorAll('input[type="checkbox"]:checked').length;
                const priceChanged = widget.querySelector('.price-slider') && 
                    (widget.querySelector('[name="filter_price_min"]').value !== widget.querySelector('[name="filter_price_min"]').min ||
                     widget.querySelector('[name="filter_price_max"]').value !== widget.querySelector('[name="filter_price_max"]').max);
                const total = checked + (priceChanged ? 1 : 0);
                
                const badge = widget.querySelector('.active-count');
                const resetBtn = widget.querySelector('.reset-filters');
                
                if (total > 0) {
                    badge.textContent = total;
                    badge.classList.remove('hidden');
                    if (resetBtn) resetBtn.classList.remove('hidden');
                    resetBtn.classList.add('flex');
                } else {
                    badge.classList.add('hidden');
                    if (resetBtn) {
                        resetBtn.classList.add('hidden');
                        resetBtn.classList.remove('flex');
                    }
                }
            }

            // Reset filters
            const resetBtn = widget.querySelector('.reset-filters');
            if (resetBtn) {
                resetBtn.addEventListener('click', () => {
                    widget.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
                    widget.querySelectorAll('.price-slider').forEach(slider => {
                        slider.value = slider.name.includes('min') ? slider.min : slider.max;
                    });
                    updateActiveCount();
                });
            }

            // Listen for changes
            widget.querySelectorAll('input').forEach(input => {
                input.addEventListener('change', updateActiveCount);
            });
        })();
        </script>
        <?php
    }
}
