<?php
/**
 * Features Showcase Widget
 *
 * Bento-style asymmetric grid layout with hero feature
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Features_Showcase extends AI_Widget_Base {

    public function get_name() {
        return 'ai_features_showcase';
    }

    public function get_title() {
        return __('Features Showcase', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-gallery-grid';
    }

    public function get_categories() {
        return array('ai-theme-features');
    }

    public function get_keywords() {
        return array('features', 'showcase', 'bento', 'grid', 'asymmetric', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'icon',
            array(
                'label'   => __('Icon', 'ai-theme'),
                'type'    => Controls_Manager::ICONS,
                'default' => array(
                    'value'   => 'fas fa-bolt',
                    'library' => 'fa-solid',
                ),
            )
        );

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'description',
            array(
                'label'   => __('Description', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'features',
            array(
                'label'   => __('Features', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $features = $settings['features'];

        // Split features: first one is hero, rest go in grid
        $hero_feature = !empty($features[0]) ? $features[0] : null;
        $grid_features = array_slice($features, 1, 3);
        ?>
        <section
            class="ai-features ai-features-showcase py-20 md:py-28"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="features"
            data-block-variant="features-showcase"
        >
            <div class="max-w-6xl mx-auto px-6">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-16">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-4xl md:text-5xl font-bold text-[var(--text-primary)] mb-4">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <!-- Bento Grid -->
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    <?php if ($hero_feature) : ?>
                        <!-- Hero Feature - Large Card -->
                        <div class="lg:col-span-2 lg:row-span-2 group relative bg-[var(--bg-secondary)] border border-[var(--border-color)] rounded-3xl p-10 md:p-12 overflow-hidden">
                            <div class="relative h-full flex flex-col justify-between min-h-[300px]">
                                <div>
                                    <!-- Icon -->
                                    <?php if (!empty($hero_feature['icon']['value'])) : ?>
                                        <div class="w-16 h-16 rounded-2xl bg-[var(--primary)] flex items-center justify-center mb-8 group-hover:scale-110 transition-transform duration-300">
                                            <i class="<?php echo esc_attr($hero_feature['icon']['value']); ?> text-3xl text-[var(--text-on-primary)]"></i>
                                        </div>
                                    <?php endif; ?>

                                    <!-- Content -->
                                    <h3 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)] mb-4 tracking-tight">
                                        <?php echo esc_html($hero_feature['title']); ?>
                                    </h3>
                                    <p class="text-lg text-[var(--text-secondary)] max-w-md leading-relaxed">
                                        <?php echo esc_html($hero_feature['description']); ?>
                                    </p>
                                </div>

                                <!-- Bottom decoration -->
                                <div class="mt-8 pt-8 border-t border-[var(--border-color)] flex items-center gap-3">
                                    <div class="w-2 h-2 rounded-full bg-[var(--primary)] animate-pulse"></div>
                                    <span class="text-sm text-[var(--text-muted)] uppercase tracking-wider">Featured</span>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- Supporting Features Grid -->
                    <?php foreach ($grid_features as $idx => $feature) : ?>
                        <div class="group relative bg-[var(--bg-primary)] rounded-3xl p-8 shadow-sm border border-[var(--border-color,#e5e7eb)] hover:shadow-xl transition-all duration-300">
                            <!-- Hover Gradient -->
                            <div class="absolute inset-0 rounded-3xl bg-gradient-to-br from-[var(--bg-secondary)] to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>

                            <div class="relative">
                                <!-- Icon -->
                                <?php if (!empty($feature['icon']['value'])) : ?>
                                    <div class="w-12 h-12 rounded-xl flex items-center justify-center mb-5 shadow-lg group-hover:scale-110 transition-transform duration-300" style="background: linear-gradient(to bottom right, var(--primary), var(--secondary)); box-shadow: 0 10px 15px -3px var(--primary-light);">
                                        <i class="<?php echo esc_attr($feature['icon']['value']); ?> text-lg text-white"></i>
                                    </div>
                                <?php endif; ?>

                                <!-- Content -->
                                <h3 class="text-xl font-semibold text-[var(--text-primary)] mb-2">
                                    <?php echo esc_html($feature['title']); ?>
                                </h3>
                                <p class="text-[var(--text-muted)] leading-relaxed">
                                    <?php echo esc_html($feature['description']); ?>
                                </p>
                            </div>

                            <!-- Index marker -->
                            <div class="absolute top-6 right-6 text-5xl font-bold text-[var(--text-primary)] opacity-10 select-none">
                                <?php echo str_pad($idx + 2, 2, '0', STR_PAD_LEFT); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
