<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) exit;

class Cart_Standard extends AI_Widget_Base {
    public function get_name() { return 'ai_cart_standard'; }
    public function get_title() { return __('Cart Standard', 'ai-theme'); }
    public function get_icon() { return 'eicon-cart'; }
    public function get_categories() { return array('ai-theme-cart'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);
        $this->add_control('title', ['label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => '']);

        $repeater = new Repeater();
        $repeater->add_control('image', ['label' => __('Image', 'ai-theme'), 'type' => Controls_Manager::MEDIA]);
        $repeater->add_control('name', ['label' => __('Name', 'ai-theme'), 'type' => Controls_Manager::TEXT]);
        $repeater->add_control('price', ['label' => __('Price', 'ai-theme'), 'type' => Controls_Manager::TEXT]);
        $repeater->add_control('quantity', ['label' => __('Quantity', 'ai-theme'), 'type' => Controls_Manager::NUMBER, 'default' => 1]);
        $this->add_control('items', ['label' => __('Items', 'ai-theme'), 'type' => Controls_Manager::REPEATER, 'fields' => $repeater->get_controls()]);

        $this->add_control('subtotal', ['label' => __('Subtotal', 'ai-theme'), 'type' => Controls_Manager::TEXT]);
        $this->add_control('shipping', ['label' => __('Shipping', 'ai-theme'), 'type' => Controls_Manager::TEXT]);
        $this->add_control('total', ['label' => __('Total', 'ai-theme'), 'type' => Controls_Manager::TEXT]);
        $this->add_control('checkout_text', ['label' => __('Checkout Button', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => '']);
        $this->end_controls_section();
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        // Get real WooCommerce cart data if available
        $use_wc_cart = class_exists('WooCommerce') && WC()->cart;
        $cart_items = [];
        $subtotal = '';
        $total = '';
        $cart_empty = true;
        $checkout_url = $s['checkout_link']['url'] ?? '/checkout';

        if ($use_wc_cart) {
            $cart = WC()->cart;
            $cart_empty = $cart->is_empty();

            if (!$cart_empty) {
                foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
                    $product = $cart_item['data'];
                    $product_id = $cart_item['product_id'];
                    $quantity = $cart_item['quantity'];

                    $image_url = '';
                    $image_id = $product->get_image_id();
                    if ($image_id) {
                        $image_url = wp_get_attachment_url($image_id);
                    }
                    if (!$image_url) {
                        $image_url = wc_placeholder_img_src();
                    }

                    $cart_items[] = [
                        'key' => $cart_item_key,
                        'product_id' => $product_id,
                        'name' => $product->get_name(),
                        'price' => $cart->get_product_price($product),
                        'quantity' => $quantity,
                        'subtotal' => $cart->get_product_subtotal($product, $quantity),
                        'image' => ['url' => $image_url],
                        'permalink' => get_permalink($product_id),
                    ];
                }

                $subtotal = $cart->get_cart_subtotal();
                $total = $cart->get_cart_total();
            }

            $checkout_url = wc_get_checkout_url();
        } else {
            // Fallback to static items from settings
            $cart_items = $s['items'] ?? [];
            $subtotal = $s['subtotal'] ?? '';
            $total = $s['total'] ?? '';
            $cart_empty = empty($cart_items);
        }
        ?>
        <div class="ai-cart ai-cart-standard w-full ai-container py-8" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="cart" data-block-variant="cart-standard">
            <h1 class="text-3xl font-bold text-[var(--text-primary)] mb-8"><?php echo esc_html($s['title']); ?></h1>

            <?php if ($cart_empty) : ?>
            <!-- Empty Cart -->
            <div class="text-center py-16">
                <div class="w-24 h-24 mx-auto mb-6 bg-[var(--bg-secondary)] rounded-full flex items-center justify-center">
                    <svg class="w-12 h-12 text-[var(--text-muted)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                    </svg>
                </div>
                <h2 class="text-2xl font-bold text-[var(--text-primary)] mb-2"><?php echo esc_html(ai_get_text('ecommerce.cart.emptyTitle', 'Your cart is empty')); ?></h2>
                <p class="text-[var(--text-secondary)] mb-6"><?php echo esc_html(ai_get_text('ecommerce.cart.emptyDescription', 'Looks like you haven\'t added any products yet.')); ?></p>
                <a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>" class="inline-flex items-center px-6 py-3 bg-primary text-white font-semibold rounded-lg hover:opacity-90 transition-opacity">
                    <?php echo esc_html(ai_get_text('ecommerce.cart.continueShopping', 'Continue Shopping')); ?>
                </a>
            </div>
            <?php else : ?>
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Cart Items -->
                <div class="ai-cart-items lg:col-span-2 space-y-4">
                    <?php foreach ($cart_items as $item) : ?>
                    <div class="ai-cart-item bg-[var(--bg-primary)] rounded-lg shadow-sm border border-[var(--border-color)] p-6 hover:shadow-md transition-shadow">
                        <div class="flex gap-4">
                            <?php if (!empty($item['image']['url'])) : ?>
                            <a href="<?php echo esc_url($item['permalink'] ?? '#'); ?>" class="flex-shrink-0 w-24 h-24 bg-[var(--bg-secondary)] rounded-lg overflow-hidden group">
                                <img src="<?php echo esc_url($item['image']['url']); ?>" alt="<?php echo esc_attr($item['name']); ?>" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500">
                            </a>
                            <?php endif; ?>

                            <div class="ai-cart-item-info flex-1 min-w-0">
                                <a href="<?php echo esc_url($item['permalink'] ?? '#'); ?>">
                                    <h3 class="text-lg font-semibold text-[var(--text-primary)] mb-1 truncate hover:text-primary transition-colors"><?php echo esc_html($item['name']); ?></h3>
                                </a>
                                <div class="mt-4 flex items-center justify-between">
                                    <?php if ($use_wc_cart && !empty($item['key'])) : ?>
                                    <div class="flex items-center gap-2">
                                        <form action="<?php echo esc_url(wc_get_cart_url()); ?>" method="post" class="flex items-center gap-2">
                                            <label class="text-sm text-[var(--text-secondary)]"><?php echo esc_html(ai_get_text('ecommerce.cart.quantity', 'Qty:')); ?></label>
                                            <input type="number" name="cart[<?php echo esc_attr($item['key']); ?>][qty]" value="<?php echo esc_attr($item['quantity']); ?>" min="0" max="999" class="w-16 px-2 py-1 border border-[var(--border-color)] rounded text-center">
                                            <button type="submit" name="update_cart" value="1" class="text-sm text-primary hover:underline"><?php echo esc_html(ai_get_text('ecommerce.cart.update', 'Update')); ?></button>
                                            <?php wp_nonce_field('woocommerce-cart', 'woocommerce-cart-nonce'); ?>
                                        </form>
                                        <a href="<?php echo esc_url(wc_get_cart_remove_url($item['key'])); ?>" class="text-red-500 hover:text-red-700 ml-4" title="<?php echo esc_attr(ai_get_text('ecommerce.cart.remove', 'Remove')); ?>">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                            </svg>
                                        </a>
                                    </div>
                                    <?php else : ?>
                                    <span class="ai-cart-quantity text-sm text-[var(--text-secondary)]"><?php echo esc_html(ai_get_text('ecommerce.cart.quantity', 'Qty:')); ?> <?php echo esc_html($item['quantity']); ?></span>
                                    <?php endif; ?>
                                    <div class="text-right">
                                        <p class="ai-cart-price text-xl font-bold text-[var(--text-primary)]"><?php echo $use_wc_cart ? $item['subtotal'] : esc_html($item['price']); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- Cart Summary -->
                <div class="lg:col-span-1">
                    <div class="ai-cart-summary bg-[var(--bg-primary)] rounded-lg shadow-sm border border-[var(--border-color)] p-6 sticky top-4">
                        <h2 class="text-xl font-bold text-[var(--text-primary)] mb-6"><?php echo esc_html(ai_get_text('ecommerce.cart.summary', 'Summary')); ?></h2>

                        <div class="space-y-3 mb-6 pb-6 border-b border-[var(--border-color)]">
                            <?php if (!empty($subtotal)) : ?>
                            <div class="ai-cart-row flex justify-between text-[var(--text-secondary)]">
                                <span><?php echo esc_html(ai_get_text('ecommerce.cart.subtotal', 'Subtotal:')); ?></span>
                                <span><?php echo $subtotal; ?></span>
                            </div>
                            <?php endif; ?>

                            <?php if ($use_wc_cart && WC()->cart->needs_shipping()) : ?>
                            <div class="ai-cart-row flex justify-between text-[var(--text-secondary)]">
                                <span><?php echo esc_html(ai_get_text('ecommerce.cart.shipping', 'Shipping:')); ?></span>
                                <span><?php echo esc_html(ai_get_text('ecommerce.cart.calculatedAtCheckout', 'Calculated at checkout')); ?></span>
                            </div>
                            <?php elseif (!empty($s['shipping'])) : ?>
                            <div class="ai-cart-row flex justify-between text-[var(--text-secondary)]">
                                <span><?php echo esc_html(ai_get_text('ecommerce.cart.shipping', 'Shipping:')); ?></span>
                                <span><?php echo esc_html($s['shipping']); ?></span>
                            </div>
                            <?php endif; ?>

                            <?php if (!empty($total)) : ?>
                            <div class="ai-cart-row ai-cart-total flex justify-between items-center pt-3">
                                <span class="text-lg font-semibold text-[var(--text-primary)]"><?php echo esc_html(ai_get_text('ecommerce.cart.total', 'Total:')); ?></span>
                                <span class="text-2xl font-bold text-[var(--text-primary)]"><?php echo $total; ?></span>
                            </div>
                            <?php endif; ?>
                        </div>

                        <a href="<?php echo esc_url($checkout_url); ?>" class="ai-cart-checkout w-full bg-primary text-white py-3 px-6 rounded-lg font-semibold hover:opacity-90 transition-opacity flex items-center justify-center gap-2"><?php echo esc_html(ai_get_text('ecommerce.cart.proceedToCheckout', '')); ?></a>

                        <a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>" class="mt-3 w-full text-center block text-[var(--text-secondary)] hover:text-primary transition-colors">
                            <?php echo esc_html(ai_get_text('ecommerce.cart.continueShopping', 'Continue Shopping')); ?>
                        </a>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
}
