<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

class Blog_List extends AI_Widget_Base {
    public function get_name() { return 'ai_blog_list'; }
    public function get_title() { return __('Blog List (WordPress Posts)', 'ai-theme'); }
    public function get_icon() { return 'eicon-post-list'; }
    public function get_categories() { return array('ai-theme-blog'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);

        $this->add_control('title', [
            'label' => __('Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ]);

        $this->add_control('posts_per_page', [
            'label' => __('Number of Posts', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 5,
            'min' => 1,
            'max' => 20,
        ]);

        $this->add_control('orderby', [
            'label' => __('Order By', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'date',
            'options' => [
                'date' => __('Date', 'ai-theme'),
                'title' => __('Title', 'ai-theme'),
                'modified' => __('Last Modified', 'ai-theme'),
                'rand' => __('Random', 'ai-theme'),
            ],
        ]);

        $categories = $this->get_post_categories();
        $this->add_control('category', [
            'label' => __('Category', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => '',
            'options' => $categories,
        ]);

        $this->add_control('show_author', [
            'label' => __('Show Author', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_date', [
            'label' => __('Show Date', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_excerpt', [
            'label' => __('Show Excerpt', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();
    }

    private function get_post_categories() {
        $categories = ['' => __('All Categories', 'ai-theme')];
        $terms = get_categories(['hide_empty' => true]);
        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[$term->slug] = $term->name;
            }
        }
        return $categories;
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $posts = $this->get_posts($s);
        ?>
        <section class="ai-blog ai-blog-list section-light py-16" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="blog" data-block-variant="blog-list">
            <div class="ai-container-narrow">
                <div class="mb-12">
                    <?php if (!empty($s['title'])) : ?><h2 class="text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($s['title']); ?></h2><?php endif; ?>
                </div>

                <?php if (empty($posts)) : ?>
                <div class="text-center py-12">
                    <p class="text-[var(--text-muted)]"><?php echo esc_html(ai_get_text('blocks.blog.noPosts', '')); ?></p>
                </div>
                <?php else : ?>
                <div class="ai-blog-posts space-y-8">
                    <?php foreach ($posts as $post) : $this->render_post($post, $s); endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }

    private function get_posts($settings) {
        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $settings['posts_per_page'] ?? 5,
            'orderby' => $settings['orderby'] ?? 'date',
            'order' => 'DESC',
        ];

        if (!empty($settings['category'])) {
            $args['category_name'] = $settings['category'];
        }

        return (new \WP_Query($args))->posts;
    }

    private function render_post($post, $settings) {
        $thumbnail_url = get_the_post_thumbnail_url($post->ID, 'medium');
        $author = get_the_author_meta('display_name', $post->post_author);
        ?>
        <article class="ai-blog-post flex gap-6 pb-8 border-b border-[var(--border-color)] last:border-b-0">
            <?php if ($thumbnail_url) : ?>
            <a href="<?php echo esc_url(get_permalink($post)); ?>" class="ai-blog-image flex-shrink-0">
                <img src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($post->post_title); ?>" class="w-48 h-32 object-cover rounded-xl">
            </a>
            <?php endif; ?>
            <div class="ai-blog-content flex-1">
                <h3 class="text-2xl font-bold text-[var(--text-primary)] mb-2">
                    <a href="<?php echo esc_url(get_permalink($post)); ?>" class="hover:text-[var(--primary)] transition-colors"><?php echo esc_html($post->post_title); ?></a>
                </h3>
                <?php if ($settings['show_excerpt'] === 'yes') : ?>
                <p class="text-[var(--text-secondary)] mb-4"><?php echo esc_html(wp_trim_words(get_the_excerpt($post), 25)); ?></p>
                <?php endif; ?>
                <div class="ai-blog-meta flex items-center gap-4 text-sm text-[var(--text-muted)]">
                    <?php if ($settings['show_author'] === 'yes') : ?>
                    <span class="ai-blog-author"><?php echo esc_html($author); ?></span>
                    <?php endif; ?>
                    <?php if ($settings['show_date'] === 'yes') : ?>
                    <span class="ai-blog-date"><?php echo esc_html(get_the_date('', $post)); ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </article>
        <?php
    }
}
