<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

class Blog_Grid extends AI_Widget_Base {
    public function get_name() { return 'ai_blog_grid'; }
    public function get_title() { return __('Blog Grid (WordPress Posts)', 'ai-theme'); }
    public function get_icon() { return 'eicon-posts-grid'; }
    public function get_categories() { return array('ai-theme-blog'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);

        $this->add_control('title', [
            'label' => __('Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ]);

        $this->add_control('subtitle', [
            'label' => __('Subtitle', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
        ]);

        $this->add_control('source', [
            'label' => __('Posts Source', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'wordpress',
            'options' => [
                'wordpress' => __('WordPress Posts', 'ai-theme'),
                'manual' => __('Manual (Custom)', 'ai-theme'),
            ],
        ]);

        // WordPress settings
        $this->add_control('posts_per_page', [
            'label' => __('Number of Posts', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 6,
            'min' => 1,
            'max' => 24,
            'condition' => ['source' => 'wordpress'],
        ]);

        $this->add_control('orderby', [
            'label' => __('Order By', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'date',
            'options' => [
                'date' => __('Date', 'ai-theme'),
                'title' => __('Title', 'ai-theme'),
                'modified' => __('Last Modified', 'ai-theme'),
                'rand' => __('Random', 'ai-theme'),
                'comment_count' => __('Comment Count', 'ai-theme'),
            ],
            'condition' => ['source' => 'wordpress'],
        ]);

        $this->add_control('order', [
            'label' => __('Order', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'DESC',
            'options' => [
                'DESC' => __('Descending', 'ai-theme'),
                'ASC' => __('Ascending', 'ai-theme'),
            ],
            'condition' => ['source' => 'wordpress'],
        ]);

        // Get post categories
        $categories = $this->get_post_categories();
        $this->add_control('category', [
            'label' => __('Category', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => '',
            'options' => $categories,
            'condition' => ['source' => 'wordpress'],
        ]);

        $this->add_control('columns', [
            'label' => __('Columns', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'options' => ['2' => '2', '3' => '3', '4' => '4'],
            'default' => '3',
        ]);

        $this->add_control('show_date', [
            'label' => __('Show Date', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_category', [
            'label' => __('Show Category', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_excerpt', [
            'label' => __('Show Excerpt', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('excerpt_length', [
            'label' => __('Excerpt Length (words)', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 20,
            'min' => 5,
            'max' => 100,
            'condition' => ['show_excerpt' => 'yes'],
        ]);

        $this->end_controls_section();
    }

    private function get_post_categories() {
        $categories = ['' => __('All Categories', 'ai-theme')];

        $terms = get_categories([
            'hide_empty' => true,
        ]);

        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[$term->slug] = $term->name;
            }
        }

        return $categories;
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        $grid_cols = [
            '2' => 'grid-cols-1 md:grid-cols-2',
            '3' => 'grid-cols-1 md:grid-cols-2 lg:grid-cols-3',
            '4' => 'grid-cols-1 md:grid-cols-2 lg:grid-cols-4'
        ];
        $grid_class = $grid_cols[$s['columns']] ?? 'grid-cols-1 md:grid-cols-2 lg:grid-cols-3';

        // Get posts from WordPress
        $posts = $this->get_posts($s);
        ?>
        <section class="ai-blog ai-blog-grid section-light py-16" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="blog" data-block-variant="blog-grid">
            <div class="ai-container">
                <?php if (!empty($s['title']) || !empty($s['subtitle'])) : ?>
                <div class="ai-blog-header text-center mb-12">
                    <?php if (!empty($s['title'])) : ?>
                    <h2 class="text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($s['title']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($s['subtitle'])) : ?>
                    <p class="text-lg text-[var(--text-secondary)]"><?php echo esc_html($s['subtitle']); ?></p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

                <?php if (empty($posts)) : ?>
                <div class="text-center py-12">
                    <p class="text-[var(--text-muted)]"><?php echo esc_html(ai_get_text('blocks.blog.noPosts', '')); ?></p>
                </div>
                <?php else : ?>
                <div class="ai-blog-posts grid <?php echo esc_attr($grid_class); ?> gap-8">
                    <?php foreach ($posts as $post) :
                        $this->render_post_card($post, $s);
                    endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }

    private function get_posts($settings) {
        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $settings['posts_per_page'] ?? 6,
            'orderby' => $settings['orderby'] ?? 'date',
            'order' => $settings['order'] ?? 'DESC',
        ];

        if (!empty($settings['category'])) {
            $args['category_name'] = $settings['category'];
        }

        $query = new \WP_Query($args);

        return $query->posts;
    }

    private function render_post_card($post, $settings) {
        $thumbnail_url = get_the_post_thumbnail_url($post->ID, 'medium_large');
        $categories = get_the_category($post->ID);
        $primary_category = !empty($categories) ? $categories[0] : null;
        $excerpt = $settings['show_excerpt'] === 'yes'
            ? wp_trim_words(get_the_excerpt($post), $settings['excerpt_length'] ?? 20)
            : '';
        ?>
        <article class="ai-blog-post bg-[var(--bg-primary)] rounded-2xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 transform hover:-translate-y-1">
            <?php if ($thumbnail_url) : ?>
            <a href="<?php echo esc_url(get_permalink($post)); ?>" class="ai-blog-image block overflow-hidden">
                <img src="<?php echo esc_url($thumbnail_url); ?>"
                     alt="<?php echo esc_attr($post->post_title); ?>"
                     class="w-full h-48 object-cover transition-transform duration-500 hover:scale-110">
            </a>
            <?php endif; ?>

            <div class="ai-blog-content p-6">
                <?php if ($settings['show_category'] === 'yes' && $primary_category) : ?>
                <a href="<?php echo esc_url(get_category_link($primary_category->term_id)); ?>"
                   class="ai-blog-category inline-block px-3 py-1 bg-[var(--primary-light)] text-[var(--primary)] text-sm font-semibold rounded-full mb-3 hover:opacity-80 transition-colors">
                    <?php echo esc_html($primary_category->name); ?>
                </a>
                <?php endif; ?>

                <h3 class="text-xl font-bold text-[var(--text-primary)] mb-2 line-clamp-2">
                    <a href="<?php echo esc_url(get_permalink($post)); ?>" class="hover:text-[var(--primary)] transition-colors">
                        <?php echo esc_html($post->post_title); ?>
                    </a>
                </h3>

                <?php if ($settings['show_excerpt'] === 'yes' && $excerpt) : ?>
                <p class="text-[var(--text-secondary)] mb-4 line-clamp-3"><?php echo esc_html($excerpt); ?></p>
                <?php endif; ?>

                <?php if ($settings['show_date'] === 'yes') : ?>
                <div class="flex items-center text-sm text-[var(--text-muted)]">
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                    </svg>
                    <time datetime="<?php echo esc_attr(get_the_date('c', $post)); ?>">
                        <?php echo esc_html(get_the_date('', $post)); ?>
                    </time>
                </div>
                <?php endif; ?>
            </div>
        </article>
        <?php
    }
}
